// hosts.js - loads hosts, shows add/edit modal, preview, delete, and loads languages
// Ready to paste to /dashboard/assets/js/hosts.js

const API = "/dashboard/api/";
const hostsBody = document.getElementById('hostsBody');
const hostModal = document.getElementById('hostModal');
const previewModal = document.getElementById('previewModal');
const previewCard = document.getElementById('previewCard');
const hostForm = document.getElementById('hostForm');
const hostLang = document.getElementById('hostLang');

let languageMap = {}; // id => name

document.addEventListener('DOMContentLoaded', initHosts);

function initHosts(){
  // buttons
  const refreshBtn = document.getElementById('refreshBtn');
  const openAddBtn = document.getElementById('openAdd');
  const closeModalBtn = document.getElementById('closeModal');
  const saveHostBtn = document.getElementById('saveHost');

  if(refreshBtn) refreshBtn.addEventListener('click', loadHosts);
  if(openAddBtn) openAddBtn.addEventListener('click', ()=>openHostModal());
  if(closeModalBtn) closeModalBtn.addEventListener('click', closeHostModal);
  if(saveHostBtn) saveHostBtn.addEventListener('click', saveHost);

  // close preview on click outside
  if(previewModal) previewModal.addEventListener('click', (e)=>{ if(e.target === previewModal) closePreview(); });

  loadLanguages().then(loadHosts).catch(err=>{
    console.error('Init error', err);
    loadHosts();
  });
}

async function loadLanguages(){
  hostLang.innerHTML = '<option value="">Select language</option>';
  languageMap = {};
  try {
    const res = await fetch(API + 'get_languages.php');
    if(!res.ok) throw new Error('langs fetch failed');
    const data = await res.json();
    if (data && Array.isArray(data.data)){
      data.data.forEach(l=>{
        languageMap[String(l.id)] = l.name;
        const opt = document.createElement('option');
        opt.value = l.id;
        opt.textContent = l.name;
        hostLang.appendChild(opt);
      });
    } else {
      // no languages available - leave the select with default
      console.warn('No languages returned from API');
    }
  } catch(e){
    console.error('Failed to load languages', e);
    // still continue, host list can use fallback values
  }
}

async function loadHosts(){
  hostsBody.innerHTML = `<tr><td colspan="4" style="text-align:center;color:var(--text-muted);padding:40px">Loading...</td></tr>`;
  try {
    const res = await fetch(API + 'get_hosts.php');
    if(!res.ok) throw new Error('hosts fetch failed: ' + res.status);
    const json = await res.json();
    if(!json || !Array.isArray(json.data) || json.data.length===0){
      hostsBody.innerHTML = `<tr><td colspan="4" style="text-align:center;color:var(--text-muted);padding:40px">No hosts found.</td></tr>`;
      return;
    }

    hostsBody.innerHTML = '';
    json.data.forEach(h=>{
      const tr = document.createElement('tr');

      // Photo cell
      const photoTD = document.createElement('td');
      const img = document.createElement('img');
      img.className = 'host-avatar';
      img.src = h.photo_url || (h.photo && '/dashboard/uploads/' + h.photo) || '/dashboard/assets/images/user-placeholder.png';
      img.alt = h.name || '';
      photoTD.appendChild(img);

      // Name cell
      const nameTD = document.createElement('td');
      nameTD.textContent = h.name || '';

      // Language cell: try language_name, else map from language_id, else empty
      const langTD = document.createElement('td');
      const langName = h.language_name || languageMap[String(h.language_id)] || '';
      langTD.textContent = langName;

      // Actions
      const actionsTD = document.createElement('td');
      actionsTD.className = 'actions-cell';
      // buttons
      const previewBtn = document.createElement('button');
      previewBtn.className = 'btn-ghost-small';
      previewBtn.textContent = 'Preview';
      previewBtn.onclick = ()=> previewHost(h.id);

      const editBtn = document.createElement('button');
      editBtn.className = 'btn-ghost-small';
      editBtn.textContent = 'Edit';
      editBtn.onclick = ()=> editHost(h.id);

      const delBtn = document.createElement('button');
      delBtn.className = 'btn-danger-small';
      delBtn.textContent = 'Delete';
      delBtn.onclick = ()=> deleteHost(h.id);

      actionsTD.appendChild(previewBtn);
      actionsTD.appendChild(editBtn);
      actionsTD.appendChild(delBtn);

      tr.appendChild(photoTD);
      tr.appendChild(nameTD);
      tr.appendChild(langTD);
      tr.appendChild(actionsTD);
      hostsBody.appendChild(tr);
    });

  } catch (e){
    console.error(e);
    hostsBody.innerHTML = `<tr><td colspan="4" style="text-align:center;color:var(--text-muted);padding:40px">Failed to load hosts.</td></tr>`;
  }
}

/* Open modal - if data provided, fills fields for edit */
function openHostModal(data){
  document.getElementById('hostId').value = data?.id || 0;
  document.getElementById('hostName').value = data?.name || '';
  document.getElementById('hostBio').value = data?.bio || '';
  if (data?.language_id) hostLang.value = data.language_id;
  else hostLang.value = '';
  hostModal.classList.add('open');
}

function closeHostModal(){
  hostModal.classList.remove('open');
  hostForm.reset();
}

/* Save host (create or update) */
async function saveHost(){
  const id = document.getElementById('hostId').value || 0;
  const name = document.getElementById('hostName').value.trim();
  const language_id = hostLang.value;
  if (!name || !language_id) { alert('Name and language are required'); return; }

  const fd = new FormData(hostForm);
  const url = API + (id && id>0 ? 'update_host.php' : 'save_host.php');

  try {
    const res = await fetch(url, { method:'POST', body: fd });
    const json = await res.json();
    if (json.ok){
      closeHostModal();
      loadHosts();
    } else {
      alert('Error saving host: ' + (json.error || JSON.stringify(json)));
    }
  } catch(e){
    console.error(e);
    alert('Error saving host.');
  }
}

/* Edit: fetch host info and open modal */
async function editHost(id){
  try {
    const res = await fetch(API + 'get_hosts.php?id=' + encodeURIComponent(id));
    const json = await res.json();
    if (json && json.data && json.data.length){
      openHostModal(json.data[0]);
    } else {
      alert('Host not found');
    }
  } catch(e){
    console.error(e);
    alert('Failed to fetch host');
  }
}

/* Delete */
async function deleteHost(id){
  if(!confirm('Delete host?')) return;
  try {
    const res = await fetch(API + 'delete_host.php', {
      method:'POST',
      headers:{ 'Content-Type':'application/json' },
      body: JSON.stringify({ id: id })
    });
    const json = await res.json();
    if(json.ok) loadHosts();
    else alert('Delete failed: ' + (json.error || 'unknown'));
  } catch(e){
    console.error(e);
    alert('Delete failed');
  }
}

/* Preview (richer UI) */
async function previewHost(id){
  try {
    const res = await fetch(API + 'get_hosts.php?id=' + encodeURIComponent(id));
    const json = await res.json();
    const h = (json && json.data && json.data[0]) || null;
    if(!h){ alert('Host not found'); return; }

    const langName = h.language_name || languageMap[String(h.language_id)] || '';
    previewCard.innerHTML = `
      <div style="display:flex;gap:14px;align-items:center;">
        <img src="${h.photo_url|| (h.photo && '/dashboard/uploads/' + h.photo) || '/dashboard/assets/images/user-placeholder.png'}" style="width:96px;height:96px;border-radius:10px;object-fit:cover;border:1px solid var(--border-subtle)">
        <div style="flex:1">
          <h3 style="margin:0 0 6px">${escapeHtml(h.name||'')}</h3>
          <div style="color:var(--text-muted);margin-bottom:8px">${escapeHtml(langName||'')}</div>
          <div style="color:var(--text-muted);font-size:14px;max-height:220px;overflow:auto">${escapeHtml(h.bio||'')}</div>
        </div>
      </div>
      <div style="text-align:right;margin-top:14px"><button class="top-action-btn refresh" onclick="closePreview()">Close</button></div>
    `;
    previewModal.classList.add('open');
  } catch(e){
    console.error(e);
    alert('Preview failed');
  }
}

function closePreview(){ previewModal.classList.remove('open'); previewCard.innerHTML = ''; }

/* Utility */
function escapeHtml(s){ if(!s) return ''; return String(s).replace(/[&<>"']/g,c=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }
