<?php
// /dashboard/api/check_auth.php
// Protect admin pages. Usage: require_once __DIR__.'/check_auth.php'; require_auth(['admin']);
header("Content-Type: application/json; charset=utf-8");

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/db.php'; // must set $pdo

// Helper: send JSON error and exit (for AJAX)
function json_unauth() {
    http_response_code(401);
    echo json_encode(['ok' => false, 'error' => 'unauth']);
    exit;
}

// Helper: redirect to login (for normal pages)
function redirect_login() {
    // if script is called from within an iframe/embedded request, JSON might be preferred.
    header('Location: /dashboard/login.html');
    exit;
}

// Validate remember token from cookie and restore session if valid.
// Returns user array on success, false on failure.
function restore_session_from_cookie($pdo) {
    if (empty($_COOKIE['bb_rem'])) return false;
    $token = $_COOKIE['bb_rem'];
    if (!is_string($token) || strlen($token) < 16) return false;

    try {
        $stmt = $pdo->prepare("SELECT * FROM users WHERE remember_token_hash IS NOT NULL LIMIT 1");
        $stmt->execute();
        // We cannot search by hash; we must fetch candidate rows and verify.
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
        foreach ($users as $u) {
            if (empty($u['remember_token_hash'])) continue;
            if (password_verify($token, $u['remember_token_hash'])) {
                // valid token — restore session
                $_SESSION['user_id'] = (int)$u['id'];
                $_SESSION['role'] = $u['role'] ?? 'editor';
                // rotate the remember token: create new token/hash and update cookie+db
                $newToken = bin2hex(random_bytes(32));
                $newHash = password_hash($newToken, PASSWORD_DEFAULT);
                $upd = $pdo->prepare("UPDATE users SET remember_token_hash = :h, last_login = NOW() WHERE id = :id");
                $upd->execute(['h'=>$newHash, 'id'=>$u['id']]);
                // set cookie (30 days), HttpOnly, Secure if https
                setcookie('bb_rem', $newToken, time() + (30*24*60*60), '/', '', isset($_SERVER['HTTPS']), true);
                return $u;
            }
        }
    } catch (PDOException $e) {
        // optionally log $e->getMessage()
        return false;
    }
    return false;
}

// Main auth enforcement function.
// $roles: null | string | array of allowed roles (e.g. 'admin' or ['admin','editor'])
// $ajax: whether to return JSON error (true) or redirect to login (false)
function require_auth($roles = null, $ajax = false) {
    global $pdo;
    // Basic sanity
    if (!isset($pdo) || !($pdo instanceof PDO)) {
        if ($ajax) json_unauth();
        redirect_login();
    }

    // 1) if session user exists, fetch user
    if (!empty($_SESSION['user_id'])) {
        try {
            $stmt = $pdo->prepare("SELECT * FROM users WHERE id = :id LIMIT 1");
            $stmt->execute(['id' => $_SESSION['user_id']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$user) {
                // session points to missing user
                session_unset();
                session_destroy();
                if ($ajax) json_unauth();
                redirect_login();
            }
        } catch (PDOException $e) {
            if ($ajax) json_unauth();
            redirect_login();
        }
    } else {
        // 2) try to restore from remember cookie
        $user = restore_session_from_cookie($pdo);
        if (!$user) {
            if ($ajax) json_unauth();
            redirect_login();
        }
    }

    // 3) check status (optional)
    if (isset($user['status']) && (int)$user['status'] !== 1) {
        // user disabled
        if ($ajax) json_unauth();
        redirect_login();
    }

    // 4) role check
    if ($roles !== null) {
        $allowed = (is_array($roles)) ? $roles : [$roles];
        $role = $user['role'] ?? '';
        if (!in_array($role, $allowed, true)) {
            // Forbidden for this role
            if ($ajax) {
                http_response_code(403);
                echo json_encode(['ok'=>false,'error'=>'forbidden']);
                exit;
            }
            // redirect to dashboard or login
            header('Location: /dashboard/index.html');
            exit;
        }
    }

    // 5) update last_login quick (optional)
    try {
        $upd = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = :id");
        $upd->execute(['id' => $user['id']]);
    } catch (PDOException $e) {
        // ignore
    }

    // Set a friendly global $current_user for pages that include this script
    $GLOBALS['current_user'] = $user;

    return true;
}
