/* loadLayout.js v2
   Inject shell, manage theme toggle (persisted), keyboard shortcut Alt+T,
   and make sure theme button reflects current state.
*/

(function () {
  'use strict';

  function createEl(tag, props = {}, children = []) {
    var el = document.createElement(tag);
    for (var k in props) {
      if (k === 'class') el.className = props[k];
      else if (k === 'html') el.innerHTML = props[k];
      else el.setAttribute(k, props[k]);
    }
    children.forEach(c => el.appendChild(c));
    return el;
  }

  function navLink(href, label, icon) {
    var a = createEl('a', { href: href, class: 'nav-link' });
    var i = createEl('span', { class: 'icon', html: icon || '•' });
    i.style.width = '22px';
    i.style.display = 'inline-block';
    var span = createEl('span', { class: 'text' });
    span.textContent = label;
    a.appendChild(i); a.appendChild(span);
    return a;
  }

  function applyActiveLinks() {
    var path = location.pathname.replace(/\/$/, '');
    document.querySelectorAll('#bb-sidebar a.nav-link').forEach(a => {
      try {
        var ahref = new URL(a.href, location.origin).pathname.replace(/\/$/, '');
        if (ahref === path) a.classList.add('active');
        else a.classList.remove('active');
      } catch (e) {}
    });
  }

  function addShell() {
    if (document.getElementById('bb-shell')) return;

    // locate page content
    var pageInner = document.getElementById('page-inner-content');
    if (!pageInner) {
      pageInner = createEl('div', { id: 'page-inner-content' });
      pageInner.innerHTML = '<div class="card"><p>Page content not found. Add &lt;div id="page-inner-content"&gt;...&lt;/div&gt; to this file.</p></div>';
    }

    // build shell
    var shell = createEl('div', { id: 'bb-shell' });

    // sidebar
    var sidebar = createEl('aside', { id: 'bb-sidebar' });
    var brand = createEl('div', { class: 'brand', html: '<div style="width:36px;height:36px;border-radius:8px;background:var(--accent);"></div><div><h1>BiggBoss Admin</h1><div style="font-size:11px;color:var(--muted)">Admin Panel</div></div>'});
    sidebar.appendChild(brand);

    var nav = createEl('nav');
    nav.appendChild(navLink('/dashboard/index.html','Dashboard','🏠'));
    nav.appendChild(navLink('/dashboard/languages.html','Languages','🌐'));
    nav.appendChild(navLink('/dashboard/hosts.html','Hosts','🎙️'));
    nav.appendChild(navLink('/dashboard/contestants.html','Contestants','🙋'));
    nav.appendChild(navLink('/dashboard/seasons.html','Seasons','📺'));
    nav.appendChild(navLink('/dashboard/broadcasts.html','Broadcasts','📡'));
    nav.appendChild(navLink('/dashboard/uploads.html','Uploads','🗂️'));
    nav.appendChild(navLink('/dashboard/settings.html','Settings','⚙️'));
    sidebar.appendChild(nav);

    // collapse toggle
    var foot = createEl('div', { class: 'mt-12' });
    var collapseBtn = createEl('button', { class: 'topbar-btn' });
    collapseBtn.textContent = 'Toggle';
    collapseBtn.title = 'Toggle sidebar';
    collapseBtn.addEventListener('click', function () {
      document.body.classList.toggle('sidebar-collapsed');
      try { localStorage.setItem('bb_sidebar_collapsed', document.body.classList.contains('sidebar-collapsed') ? '1' : '0'); } catch (e) {}
    });
    foot.appendChild(collapseBtn);
    sidebar.appendChild(foot);

    // topbar
    var topbar = createEl('header', { id: 'bb-topbar' });
    var left = createEl('div', { class: 'topbar-left' });
    var menuBtn = createEl('button', { class: 'topbar-btn' , title: 'Open menu' });
    menuBtn.innerText = '☰';
    menuBtn.addEventListener('click', function () {
      if (window.innerWidth <= 720) sidebar.classList.toggle('open');
    });
    left.appendChild(menuBtn);
    var pageTitle = createEl('div', { html: '<strong class="page-title">'+(document.title || 'Dashboard')+'</strong>'});
    left.appendChild(pageTitle);

    var right = createEl('div', { class: 'topbar-right' });
    // theme button (shows moon or sun)
    var themeBtn = createEl('button', { class: 'topbar-btn', id: 'bb-theme-toggle', title: 'Toggle theme (Alt+T)' });
    themeBtn.textContent = '🌙';
    right.appendChild(themeBtn);

    var userBtn = createEl('button', { class: 'topbar-btn' });
    userBtn.textContent = 'admin';
    userBtn.addEventListener('click', function () { location.href = '/dashboard/settings.html'; });
    right.appendChild(userBtn);

    topbar.appendChild(left); topbar.appendChild(right);

    // content
    var content = createEl('main', { id: 'bb-content' });
    var container = createEl('div', { class: 'container' });
    container.appendChild(pageInner);
    content.appendChild(container);

    // assemble
    shell.appendChild(sidebar);
    shell.appendChild(topbar);
    shell.appendChild(content);

    // preserve external script tags (external only) so deferred scripts still load
    var body = document.body;
    var externalScripts = Array.from(body.querySelectorAll('script[src]'));
    // move shell into body (clear body first)
    while (body.firstChild) body.removeChild(body.firstChild);
    body.appendChild(shell);
    // re-add external scripts deferred (so they run after shell)
    externalScripts.forEach(s => {
      var ns = document.createElement('script');
      ns.src = s.src;
      ns.defer = true;
      document.body.appendChild(ns);
    });

    // restore stored theme/sidebar state
    function initPrefs() {
      try {
        var theme = localStorage.getItem('bb_theme');
        if (theme === 'light') {
          document.body.classList.add('light-theme');
          themeBtn.textContent = '☀️';
        } else {
          document.body.classList.remove('light-theme');
          themeBtn.textContent = '🌙';
        }
        var sc = localStorage.getItem('bb_sidebar_collapsed');
        if (sc === '1') document.body.classList.add('sidebar-collapsed');
      } catch (e) {}
    }
    initPrefs();

    // theme toggle handler
    themeBtn.addEventListener('click', function () {
      document.body.classList.toggle('light-theme');
      var isLight = document.body.classList.contains('light-theme');
      themeBtn.textContent = isLight ? '☀️' : '🌙';
      try { localStorage.setItem('bb_theme', isLight ? 'light' : 'dark'); } catch (e) {}
    });

    // keyboard shortcut Alt+T to toggle theme
    document.addEventListener('keydown', function (ev) {
      if (ev.altKey && (ev.key === 't' || ev.key === 'T')) {
        ev.preventDefault();
        themeBtn.click();
      }
    });

    // click outside to close mobile sidebar
    document.addEventListener('click', function (ev) {
      if (window.innerWidth <= 720) {
        if (!sidebar.contains(ev.target) && !menuBtn.contains(ev.target)) sidebar.classList.remove('open');
      }
    });

    // active nav highlight & title sync
    applyActiveLinks();
    var titleObserver = new MutationObserver(() => pageTitle.innerHTML = '<strong class="page-title">'+(document.title || '')+'</strong>');
    var titleNode = document.querySelector('title');
    if (titleNode) titleObserver.observe(titleNode, { childList: true });

    // expose debug handle
    window.__bb = { sidebar, topbar, content, themeBtn };
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', addShell);
  } else addShell();
})();
