/* uploads.js
   - file manager
   - upload multiple files
   - list / preview / download / delete / rename (if API present)
   - flexible parsing of API responses
*/

(() => {
  const id = s => document.getElementById(s);
  const API = {
    list: '/dashboard/api/list_uploads.php',
    upload: '/dashboard/api/upload_file.php',
    delete: '/dashboard/api/delete_upload.php',
    rename: '/dashboard/api/rename_upload.php', // optional
    download: '/dashboard/api/download.php' // optional: ?f=encoded_path
  };

  const uploadsContainer = id('uploadsContainer');
  const refreshBtn = id('refreshBtn');
  const fileInput = id('fileInput');
  const filterType = id('filterType');
  const filterDate = id('filterDate');
  const globalSearch = id('globalSearch');
  const totalFiles = id('totalFiles');
  const lastSync = id('lastSync');

  // modal
  const modalRoot = id('modalRoot');
  const fileModal = id('fileModal');
  const fileNameEl = id('fileName');
  const fileMeta = id('fileMeta');
  const filePreviewArea = id('filePreviewArea');
  const downloadBtn = id('downloadBtn');
  const deleteBtn = id('deleteBtn');
  const renameBtn = id('renameBtn');
  const closeModalBtn = id('closeModalBtn');

  const renameArea = id('renameArea');
  const renameInput = id('renameInput');
  const renameSave = id('renameSave');
  const renameCancel = id('renameCancel');

  let cachedFiles = [];
  let activeFile = null;

  function esc(s){ return (s||'').toString().replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }

  function humanSize(bytes){
    if(!bytes) return '0 B';
    const units = ['B','KB','MB','GB','TB'];
    let i = 0;
    let b = Number(bytes);
    while(b >= 1024 && i < units.length-1){ b = b/1024; i++; }
    return `${b.toFixed(b<10 && i>0 ? 2 : 1)} ${units[i]}`;
  }

  function fileTypeFromMime(mime, name){
    if(!mime) {
      const ext = (name||'').split('.').pop().toLowerCase();
      if(['png','jpg','jpeg','gif','webp','svg'].includes(ext)) return 'image';
      if(['mp4','webm','mov','mkv'].includes(ext)) return 'video';
      if(['mp3','wav','ogg','m4a'].includes(ext)) return 'audio';
      if(['pdf','doc','docx','xls','xlsx','ppt','pptx','txt','csv','json'].includes(ext)) return 'document';
      return 'other';
    }
    if(mime.startsWith('image/')) return 'image';
    if(mime.startsWith('video/')) return 'video';
    if(mime.startsWith('audio/')) return 'audio';
    if(mime === 'application/pdf' || mime.startsWith('text/') || mime.includes('officedocument')) return 'document';
    return 'other';
  }

  // render files grid / list
  function renderFiles(files){
    cachedFiles = files || [];
    uploadsContainer.innerHTML = '';
    if(!files || files.length === 0){
      uploadsContainer.innerHTML = '<div class="recent-empty">No files uploaded yet. Click + Upload to add files.</div>';
      totalFiles.textContent = '0';
      return;
    }
    totalFiles.textContent = files.length;

    files.forEach(f => {
      const item = document.createElement('div');
      item.className = 'recent-item';
      item.style.display='flex';
      item.style.alignItems='center';
      item.style.justifyContent='space-between';
      item.style.gap='12px';

      const left = document.createElement('div');
      left.style.display='flex';
      left.style.gap='12px';
      left.style.alignItems='center';

      const thumb = document.createElement('div');
      thumb.style.width = '88px';
      thumb.style.height = '64px';
      thumb.style.borderRadius = '8px';
      thumb.style.overflow = 'hidden';
      thumb.style.display='flex';
      thumb.style.alignItems='center';
      thumb.style.justifyContent='center';
      thumb.style.background = 'var(--panel)';
      thumb.style.border = '1px solid rgba(255,255,255,0.03)';

      if(fileTypeFromMime(f.type, f.name) === 'image' && f.url){
        const img = document.createElement('img');
        img.src = f.url; img.style.width='100%'; img.style.height='100%'; img.style.objectFit='cover';
        thumb.appendChild(img);
      } else {
        thumb.innerHTML = `<div style="text-align:center;color:var(--muted);font-size:12px;">${esc((f.type||f.name||'').split('/').pop()||'file')}</div>`;
      }

      const info = document.createElement('div');
      info.style.maxWidth = '720px';
      info.innerHTML = `<div style="font-weight:700">${esc(f.name || f.filename || '')}</div>
        <div style="font-size:13px;color:var(--muted)">${esc(f.type || '')} • ${humanSize(f.size || f.file_size || 0)} • ${esc(f.created_at || '')}</div>`;

      left.appendChild(thumb);
      left.appendChild(info);

      const right = document.createElement('div');
      right.style.display='flex';
      right.style.gap='8px';

      const previewBtn = document.createElement('button'); previewBtn.className='btn small'; previewBtn.textContent='Preview';
      previewBtn.addEventListener('click', ()=> openFileModal(f));

      const downloadBtn = document.createElement('a'); downloadBtn.className='btn small'; downloadBtn.textContent='Download';
      downloadBtn.href = f.url || (API.download + '?f=' + encodeURIComponent(f.path || f.filename || '')); downloadBtn.target = '_blank';

      const deleteBtn = document.createElement('button'); deleteBtn.className='btn small'; deleteBtn.style.background='rgba(220,60,60,0.9)'; deleteBtn.textContent='Delete';
      deleteBtn.addEventListener('click', ()=> {
        if(!confirm('Delete this file?')) return;
        deleteFile(f.id || f.path || f.filename);
      });

      right.appendChild(previewBtn);
      right.appendChild(downloadBtn);
      right.appendChild(deleteBtn);

      item.appendChild(left);
      item.appendChild(right);
      uploadsContainer.appendChild(item);
    });
  }

  // open modal for file preview / actions
  function openFileModal(file){
    activeFile = file;
    fileNameEl.textContent = file.name || file.filename || 'File';
    fileMeta.textContent = `${file.type || ''} • ${humanSize(file.size || file.file_size || 0)} • ${file.created_at || ''}`;
    filePreviewArea.innerHTML = '';
    // preview by file type
    const t = fileTypeFromMime(file.type, file.name);
    if(t === 'image' && file.url){
      const img = document.createElement('img'); img.src = file.url; img.style.maxWidth='100%'; img.style.maxHeight='60vh'; img.style.borderRadius='8px';
      filePreviewArea.appendChild(img);
    } else if(t === 'video' && file.url){
      const video = document.createElement('video'); video.src = file.url; video.controls = true; video.style.maxWidth='100%'; video.style.maxHeight='60vh';
      filePreviewArea.appendChild(video);
    } else if(t === 'audio' && file.url){
      const audio = document.createElement('audio'); audio.src = file.url; audio.controls = true; filePreviewArea.appendChild(audio);
    } else if((file.type || '').includes('pdf') && file.url){
      const iframe = document.createElement('iframe'); iframe.src = file.url; iframe.style.width='100%'; iframe.style.height='70vh'; filePreviewArea.appendChild(iframe);
    } else {
      filePreviewArea.innerHTML = `<div style="color:var(--muted)">Preview not available. Use Download to get the file.</div>`;
    }

    modalRoot.style.display = 'block';
    fileModal.style.display = 'block';
  }

  closeModalBtn.addEventListener('click', () => { modalRoot.style.display='none'; fileModal.style.display='none'; renameArea.style.display='none'; });
  deleteBtn.addEventListener('click', () => {
    if(!activeFile) return;
    if(!confirm('Delete this file?')) return;
    deleteFile(activeFile.id || activeFile.path || activeFile.filename);
  });

  renameBtn.addEventListener('click', () => {
    if(!activeFile) return;
    renameArea.style.display = 'block';
    renameInput.value = activeFile.name || activeFile.filename || '';
  });

  renameCancel.addEventListener('click', () => { renameArea.style.display='none'; });

  renameSave.addEventListener('click', async () => {
    const newName = (renameInput.value || '').trim();
    if(!newName) return alert('Enter new name');
    if(!API.rename){
      alert('Rename API not configured on server.');
      return;
    }
    try {
      renameSave.disabled = true;
      const res = await fetch(API.rename, {
        method: 'POST',
        headers: {'Content-Type':'application/json'},
        body: JSON.stringify({ id: activeFile.id || activeFile.path || activeFile.filename, new_name: newName })
      });
      const json = await res.json();
      if(json.status && json.status === 'error') throw new Error(json.message || 'rename failed');
      // optimistic update
      activeFile.name = newName;
      fileNameEl.textContent = newName;
      renameArea.style.display='none';
      await refreshList();
    } catch(e){
      alert('Rename failed: ' + (e.message || 'error'));
      console.error(e);
    } finally { renameSave.disabled = false; }
  });

  // delete file
  async function deleteFile(id){
    try {
      const res = await fetch(API.delete, {
        method: 'POST',
        headers: {'Content-Type':'application/json'},
        body: JSON.stringify({ id })
      });
      const json = await res.json();
      if(json.status && json.status === 'error') throw new Error(json.message || 'delete failed');
      modalRoot.style.display='none'; fileModal.style.display='none'; renameArea.style.display='none';
      await refreshList();
    } catch(e){
      alert('Delete failed: ' + (e.message || 'error'));
      console.error(e);
    }
  }

  // upload files
  fileInput.addEventListener('change', async (e) => {
    const files = Array.from(e.target.files || []);
    if(!files.length) return;
    // upload sequentially to avoid server overload; show a progress UI simple via alerts (keeps code simple)
    for(const f of files){
      try {
        const fd = new FormData();
        fd.append('file', f);
        // optional extra fields: folder, visibility etc.
        const res = await fetch(API.upload, { method: 'POST', body: fd });
        if(!res.ok) throw new Error('HTTP ' + res.status);
        const json = await res.json();
        if(json.status && json.status === 'error') throw new Error(json.message || 'upload failed');
        // continue
      } catch(err){
        alert('Upload failed for ' + f.name + ': ' + (err.message || 'error'));
        console.error('upload error', err);
      }
    }
    // refresh after all
    await refreshList();
    fileInput.value = '';
  });

  // fetch list
  async function refreshList(){
    uploadsContainer.innerHTML = '<div class="recent-empty">Loading files...</div>';
    try {
      const r = await fetch(API.list, {cache:'no-store'});
      if(!r.ok) throw new Error('HTTP ' + r.status);
      const json = await r.json();
      const files = Array.isArray(json) ? json : (json.data || []);
      renderFiles(files);
      lastSync.textContent = new Date().toLocaleString();
    } catch(e){
      uploadsContainer.innerHTML = `<div class="recent-empty">Failed to load files: ${e.message}</div>`;
      console.error('refreshList', e);
    }
  }

  // filters & search
  filterType.addEventListener('change', applyFilters);
  filterDate.addEventListener('change', applyFilters);
  globalSearch.addEventListener('input', applyFilters);

  function applyFilters(){
    const type = (filterType.value || '').toLowerCase();
    const days = parseInt(filterDate.value || 0);
    const q = (globalSearch.value || '').toLowerCase().trim();
    const now = Date.now();

    const filtered = (cachedFiles || []).filter(f => {
      if(type){
        const ft = fileTypeFromMime(f.type, f.name);
        if(ft !== type) return false;
      }
      if(days){
        const created = new Date(f.created_at || f.uploaded_at || f.time || 0).getTime();
        if(isNaN(created)) return false;
        if((now - created) > days * 24 * 3600 * 1000) return false;
      }
      if(q){
        const hay = ((f.name||'') + ' ' + (f.type||'') + ' ' + (f.path||'')).toLowerCase();
        if(!hay.includes(q)) return false;
      }
      return true;
    });

    renderFiles(filtered);
  }

  // init
  document.addEventListener('DOMContentLoaded', refreshList);
  refreshBtn.addEventListener('click', refreshList);

})();
