/* seasons.js
   - list / view / create / edit / delete seasons
   - handles images (logo, trophy, house photos) with preview and optional upload
   - placements auto-fill from contestants
*/

(() => {
  const id = s => document.getElementById(s);
  const API = {
    list: '/dashboard/api/get_seasons.php',
    create: '/dashboard/api/save_season.php',
    update: '/dashboard/api/update_season.php',
    remove: '/dashboard/api/delete_season.php',
    uploadImage: '/dashboard/api/upload_image.php',
    hosts: '/dashboard/api/get_hosts.php',
    languages: '/dashboard/api/get_languages.php',
    contestants: '/dashboard/api/get_contestants.php'
  };

  // elements
  const container = id('seasonsContainer');
  const refreshBtn = id('refreshBtn');
  const addBtn = id('addBtn');
  const lastSync = id('lastSync');
  const filterLanguage = id('filterLanguage');
  const filterShowType = id('filterShowType');
  const searchInput = id('searchInput');
  const globalSearch = id('globalSearch');

  // view modal
  const viewModalRoot = id('viewModalRoot');
  const viewModal = id('viewModal');
  const closeView = id('closeView');
  const view_media_img = id('view_media_img');
  const view_title = id('view_title');
  const view_meta_row = id('view_meta_row');
  const view_description = id('view_description');
  const view_placements = id('view_placements');
  const view_house = id('view_house');
  const view_notes = id('view_notes');

  // edit modal elements
  const modalRoot = id('modalRoot');
  const seasonModal = id('seasonModal');
  const modalTitle = id('modalTitle');
  const m_show_type = id('m_show_type');
  const m_show_custom_wrap = id('m_show_custom_wrap');
  const m_show_custom = id('m_show_custom');
  const m_language = id('m_language');
  const m_season_no = id('m_season_no');
  const m_tag = id('m_tag');
  const m_year = id('m_year');
  const m_host = id('m_host');
  const m_notes = id('m_notes');
  const m_trophy_input = id('m_trophy_input');
  const m_trophy_preview = id('m_trophy_preview');
  const m_logo_input = id('m_logo_input');
  const m_logo_preview = id('m_logo_preview');
  const m_house_input = id('m_house_input');
  const m_house_gallery = id('m_house_gallery');
  const b_ott_name = id('b_ott_name');
  const b_ott_link = id('b_ott_link');

  const p_winner = id('p_winner');
  const p_ru1 = id('p_ru1');
  const p_ru2 = id('p_ru2');
  const p_ru3 = id('p_ru3');
  const p_ru4 = id('p_ru4');
  const autoFillBtn = id('autoFillPlacements');

  const modalCancel = id('modalCancel');
  const modalSave = id('modalSave');

  let editingId = null;
  let cachedSeasons = [];
  let contestantsList = [];

  // helpers
  function esc(s){ return (s||'').toString().replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }

  // load supporting lists
  async function loadLanguages(){
    try {
      const r = await fetch(API.languages,{cache:'no-store'});
      const json = await r.json();
      const arr = Array.isArray(json) ? json : (json.data || []);
      m_language.innerHTML = '<option value="">-- Select language --</option>' + arr.map(l=>`<option value="${esc(l.name)}">${esc(l.name)}</option>`).join('');
      filterLanguage.innerHTML = '<option value="">All languages</option>' + arr.map(l=>`<option value="${esc(l.name)}">${esc(l.name)}</option>`).join('');
    } catch(e){
      console.warn('loadLanguages',e);
    }
  }

  async function loadHosts(){
    try {
      const r = await fetch(API.hosts,{cache:'no-store'});
      const json = await r.json();
      const arr = Array.isArray(json) ? json : (json.data || []);
      m_host.innerHTML = '<option value="">-- Select host --</option>' + arr.map(h=>`<option value="${h.id}">${esc(h.name)}</option>`).join('');
    } catch(e){
      console.warn('loadHosts', e);
    }
  }

  async function loadContestants(){
    try {
      const r = await fetch(API.contestants,{cache:'no-store'});
      const json = await r.json();
      contestantsList = Array.isArray(json) ? json : (json.data || []);
      populatePlacementSelects([], contestantsList);
    } catch(e){
      console.warn('loadContestants', e);
      contestantsList = [];
    }
  }

  // render seasons list
  function renderList(items){
    cachedSeasons = items || [];
    container.innerHTML = '';
    if(!items || items.length === 0){
      container.innerHTML = '<div class="recent-empty">No seasons found. Click + Create Season to add.</div>';
      return;
    }

    items.forEach(s => {
      const row = document.createElement('div');
      row.className = 'season-card';
      const thumb = s.logo || s.trophy || (s.house_photos && s.house_photos[0]) || '';
      const hostName = s.host_name || '';
      const st = (s.show_type || 'season').toLowerCase();

      let badge = '';
      if(st === 'season') badge = '<span class="show-type-badge show-type-season">Season</span>';
      else if(st === 'ott') badge = '<span class="show-type-badge show-type-ott">OTT</span>';
      else if(st === 'ultimate') badge = '<span class="show-type-badge show-type-ultimate">Ultimate</span>';
      else badge = `<span class="show-type-badge show-type-custom">${esc(s.show_type_label || s.show_type || st)}</span>`;

      row.innerHTML = `
        <div style="display:flex;gap:12px;align-items:flex-start; width:100%;">
          <div style="width:88px;height:88px;border-radius:10px;overflow:hidden;border:1px solid rgba(255,255,255,0.03);background:var(--panel); flex:0 0 88px;">
            ${thumb ? `<img src="${thumb}" style="width:100%;height:100%;object-fit:cover;">` : ''}
          </div>

          <div style="flex:1;">
            <div style="display:flex;justify-content:space-between;align-items:center;">
              <div>
                <div style="font-weight:700;font-size:16px">${esc(s.tag || ('Season ' + (s.season_no || '-')))} ${badge}</div>
                <div style="color:var(--muted);font-size:13px;margin-top:6px">Language: ${esc(s.language||'-')} • Season No: ${esc(s.season_no||'-')} • Host: ${esc(hostName||'-')}</div>
              </div>
              <div style="display:flex;gap:8px;align-items:center;">
                <button class="btn small viewBtn">View</button>
                <button class="btn small editBtn">Edit</button>
                <button class="btn small" style="background:rgba(220,60,60,0.9)" >Delete</button>
              </div>
            </div>
          </div>
        </div>
      `;

      // wire buttons
      const viewBtn = row.querySelector('.viewBtn');
      const editBtn = row.querySelector('.editBtn');
      const delBtn = row.querySelector('button[style*="background:rgba(220,60,60"]');

      viewBtn.addEventListener('click', ()=> openViewModal(s.id));
      editBtn.addEventListener('click', ()=> openEditSeason(s.id));
      delBtn.addEventListener('click', ()=> {
        if(!confirm('Delete this season?')) return;
        deleteSeason(s.id);
      });

      container.appendChild(row);
    });
  }

  // fetch seasons
  async function fetchSeasons(){
    container.innerHTML = '<div class="recent-empty">Loading seasons...</div>';
    try {
      const r = await fetch(API.list,{cache:'no-store'});
      if(!r.ok) throw new Error(`HTTP ${r.status}`);
      const json = await r.json();
      const rows = Array.isArray(json) ? json : (json.data || []);
      renderList(rows);
      lastSync.textContent = new Date().toLocaleString();
      cachedSeasons = rows;
    } catch(e){
      container.innerHTML = `<div class="recent-empty">Unable to load seasons: ${e.message}</div>`;
      console.error('fetchSeasons', e);
    }
  }

  // open view modal
  function openViewModal(id){
    const s = cachedSeasons.find(x=>x.id == id);
    if(!s) return alert('Season not found');
    const thumb = s.logo || s.trophy || (s.house_photos && s.house_photos[0]) || '';
    if(thumb){ view_media_img.src = thumb; view_media_img.style.display='block'; } else { view_media_img.style.display='none'; }

    const label = s.show_type_label || s.show_type || 'season';
    view_title.textContent = `${s.tag || ('Season ' + (s.season_no || '-'))} (${label})`;
    view_meta_row.innerHTML = `
      <div class="chip">Language: <strong>${esc(s.language||'-')}</strong></div>
      <div class="chip">Season No: <strong>${esc(s.season_no||'-')}</strong></div>
      <div class="chip">Year: <strong>${esc(s.year||'-')}</strong></div>
      <div class="chip">Type: <strong>${esc(label||'-')}</strong></div>
      <div class="chip">Participants: <strong>${(s.participants||[]).length || 0}</strong></div>
    `;
    view_description.textContent = s.notes || '';

    // placements
    view_placements.innerHTML = '';
    const placements = s.placements || {};
    const keys = [
      {k:'winner', label:'Winner'},
      {k:'ru1', label:'1st Runner-up'},
      {k:'ru2', label:'2nd Runner-up'},
      {k:'ru3', label:'3rd Runner-up'},
      {k:'ru4', label:'4th Runner-up'}
    ];
    keys.forEach(p=>{
      const box = document.createElement('div'); box.className = 'placement-box';
      const name = findContestantName(placements[p.k]) || '-';
      box.innerHTML = `<h5>${p.label}</h5><p>${esc(name)}</p>`;
      view_placements.appendChild(box);
    });

    // house photos
    view_house.innerHTML = '';
    (s.house_photos || []).forEach(h=>{
      const img = document.createElement('img');
      img.src = h; img.style.width='100px'; img.style.height='70px'; img.style.objectFit='cover'; img.style.borderRadius='8px'; img.style.border='1px solid rgba(255,255,255,0.04)'; img.style.marginRight='8px';
      view_house.appendChild(img);
    });

    view_notes.innerHTML = `<strong>Notes</strong><div style="margin-top:6px;">${esc(s.notes || '-')}</div>`;

    viewModalRoot.style.display = 'block';
    viewModal.style.display = 'block';
  }

  closeView.addEventListener('click', ()=> { viewModalRoot.style.display='none'; viewModal.style.display='none'; });

  function findContestantName(id){
    if(!id) return '';
    const found = contestantsList.find(c=> (c.id || c.uid || c.name) == id);
    return found ? found.name : id;
  }

  // Edit / Create flow
  function openCreate(){
    editingId = null;
    modalTitle.textContent = 'Create Season';
    m_show_type.value = 'season'; m_show_custom.value = ''; m_show_custom_wrap.style.display='none';
    m_language.value = ''; m_season_no.value = ''; m_tag.value = ''; m_year.value = ''; m_notes.value = '';
    m_logo_preview.style.display = 'none'; m_trophy_preview.style.display = 'none'; m_house_gallery.innerHTML = '';
    b_ott_name.value = ''; b_ott_link.value = '';
    populatePlacementSelects([], contestantsList);
    modalRoot.style.display = 'block'; seasonModal.style.display = 'block';
  }

  async function openEditSeason(id){
    const arr = cachedSeasons;
    const s = arr.find(x=>x.id == id);
    if(!s) return alert('Season not found');
    editingId = id;
    modalTitle.textContent = 'Edit Season';

    // show_type
    const st = (s.show_type || 'season').toString().toLowerCase();
    if(['season','ott','ultimate'].includes(st)){ m_show_type.value = st; m_show_custom_wrap.style.display='none'; m_show_custom.value=''; }
    else { m_show_type.value = 'custom'; m_show_custom_wrap.style.display='block'; m_show_custom.value = s.show_type_label || s.show_type || st; }

    m_language.value = s.language || '';
    m_season_no.value = s.season_no || '';
    m_tag.value = s.tag || '';
    m_year.value = s.year || '';
    m_notes.value = s.notes || '';
    b_ott_name.value = s.broadcasts?.ott?.name || '';
    b_ott_link.value = s.broadcasts?.ott?.link || '';
    // images
    if(s.trophy){ m_trophy_preview.src = s.trophy; m_trophy_preview.style.display='block'; } else m_trophy_preview.style.display='none';
    if(s.logo){ m_logo_preview.src = s.logo; m_logo_preview.style.display='block'; } else m_logo_preview.style.display='none';
    m_house_gallery.innerHTML = '';
    (s.house_photos || []).forEach(h => {
      const wrapper = document.createElement('div'); wrapper.style.position='relative'; wrapper.style.display='inline-block'; wrapper.style.marginRight='8px';
      const img = document.createElement('img'); img.src = h; img.style.width='72px'; img.style.height='72px'; img.style.objectFit='cover'; img.style.borderRadius='8px';
      wrapper.appendChild(img); m_house_gallery.appendChild(wrapper);
    });

    // participants -> selected
    const sel = (s.participants || []).slice();
    populatePlacementSelects([s.placements?.winner, s.placements?.ru1, s.placements?.ru2, s.placements?.ru3, s.placements?.ru4], contestantsList);
    // set values for placement selects
    p_winner.value = s.placements?.winner || '';
    p_ru1.value = s.placements?.ru1 || '';
    p_ru2.value = s.placements?.ru2 || '';
    p_ru3.value = s.placements?.ru3 || '';
    p_ru4.value = s.placements?.ru4 || '';

    modalRoot.style.display = 'block'; seasonModal.style.display = 'block';
  }

  // upload helpers
  async function uploadFile(file){
    if(!file) return null;
    try{
      const fd = new FormData(); fd.append('image', file);
      const r = await fetch(API.uploadImage, { method:'POST', body: fd });
      if(!r.ok) throw new Error('Upload failed');
      const json = await r.json();
      if(json.status && json.status === 'error') throw new Error(json.message || 'upload error');
      return json.url || json.data?.url || json.path || null;
    } catch(e){
      console.warn('uploadFile', e);
      throw e;
    }
  }

  // image preview handlers
  m_logo_input.addEventListener('change', (e)=>{
    const f = e.target.files && e.target.files[0];
    if(!f){ m_logo_preview.style.display='none'; return; }
    m_logo_preview.src = URL.createObjectURL(f); m_logo_preview.style.display='block';
  });
  m_trophy_input.addEventListener('change', (e)=>{
    const f = e.target.files && e.target.files[0];
    if(!f){ m_trophy_preview.style.display='none'; return; }
    m_trophy_preview.src = URL.createObjectURL(f); m_trophy_preview.style.display='block';
  });
  m_house_input.addEventListener('change', (e)=>{
    const files = Array.from(e.target.files || []);
    files.forEach(f=>{
      const r = URL.createObjectURL(f);
      const wrapper = document.createElement('div'); wrapper.style.position='relative'; wrapper.style.display='inline-block'; wrapper.style.marginRight='8px';
      const img = document.createElement('img'); img.src = r; img.style.width='72px'; img.style.height='72px'; img.style.objectFit='cover'; img.style.borderRadius='8px';
      wrapper.appendChild(img); m_house_gallery.appendChild(wrapper);
    });
  });

  // populate placements selects
  function populatePlacementSelects(selectedIds, contestants){
    const list = contestants || contestantsList || [];
    list.sort((a,b) => (a.name||'').localeCompare(b.name||''));
    const entries = [{id:'',name:'-- None --'}].concat(list.map(c => ({id: c.id || c.uid || c.name, name: c.name || c.id || c.uid})));
    [p_winner,p_ru1,p_ru2,p_ru3,p_ru4].forEach(sel=>{
      sel.innerHTML = '';
      entries.forEach(e=>{
        const opt = document.createElement('option'); opt.value = e.id; opt.textContent = e.name; sel.appendChild(opt);
      });
    });
    if(selectedIds && selectedIds.length){
      p_winner.value = selectedIds[0] || '';
      p_ru1.value = selectedIds[1] || '';
      p_ru2.value = selectedIds[2] || '';
      p_ru3.value = selectedIds[3] || '';
      p_ru4.value = selectedIds[4] || '';
    }
  }

  autoFillBtn.addEventListener('click', ()=>{
    const arr = (contestantsList || []).slice(0,5).map(c => c.id || c.uid || c.name);
    if(!arr.length) return alert('No contestants to auto-fill.');
    p_winner.value = arr[0] || '';
    p_ru1.value = arr[1] || '';
    p_ru2.value = arr[2] || '';
    p_ru3.value = arr[3] || '';
    p_ru4.value = arr[4] || '';
  });

  // save season
  modalSave.addEventListener('click', async ()=>{
    const raw_show_type = (m_show_type.value || 'season').trim();
    const customLabel = (m_show_custom.value || '').trim();
    let show_type = raw_show_type;
    let show_type_label = '';
    if(raw_show_type === 'custom'){
      if(!customLabel) return alert('Enter custom show name');
      show_type_label = customLabel;
      show_type = customLabel.toLowerCase().replace(/\s+/g,'-');
    } else {
      show_type_label = (raw_show_type === 'season' ? 'BiggBoss Season' : (raw_show_type === 'ott' ? 'BiggBoss OTT' : (raw_show_type === 'ultimate' ? 'BiggBoss Ultimate' : raw_show_type)));
      show_type = raw_show_type;
    }

    const language = m_language.value || '';
    const season_no = m_season_no.value || '';
    const tag = m_tag.value || '';
    const year = m_year.value || '';
    const notes = m_notes.value || '';
    const host_id = m_host.value || '';
    const ott_name = b_ott_name.value || '';
    const ott_link = b_ott_link.value || '';

    if(!language) return alert('Select language');
    if(!tag) return alert('Enter season tag/title');

    modalSave.disabled = true;

    try {
      // handle uploads if files provided; else keep existing urls if editing
      let logoUrl = null;
      let trophyUrl = null;
      let houseUrls = [];

      if(m_logo_input.files && m_logo_input.files[0]) {
        try { logoUrl = await uploadFile(m_logo_input.files[0]); } catch(e){ if(!confirm('Logo upload failed. Continue without logo?')) { modalSave.disabled=false; return; } }
      }
      if(m_trophy_input.files && m_trophy_input.files[0]) {
        try { trophyUrl = await uploadFile(m_trophy_input.files[0]); } catch(e){ if(!confirm('Trophy upload failed. Continue?')) { modalSave.disabled=false; return; } }
      }
      if(m_house_input.files && m_house_input.files.length){
        for(const f of Array.from(m_house_input.files)){ try { const u = await uploadFile(f); if(u) houseUrls.push(u); } catch(e){ /* ignore individual failure */ } }
      }

      const placements = { winner: p_winner.value || '', ru1: p_ru1.value || '', ru2: p_ru2.value || '', ru3: p_ru3.value || '', ru4: p_ru4.value || '' };
      const participants = []; // this UI doesn't select participants checkbox list; leave empty or adapt if needed

      const payload = {
        show_type, show_type_label, language, season_no, tag, year, notes,
        host_id, broadcasts: { ott: ott_name || ott_link ? {name: ott_name, link: ott_link, logo: ''} : null, satellite: null },
        placements,
        participants,
        logo: logoUrl || undefined,
        trophy: trophyUrl || undefined,
        house_photos: houseUrls.length ? houseUrls : undefined
      };

      if(editingId){
        payload.id = editingId;
        const r = await fetch(API.update, { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) });
        const j = await r.json();
        if(j.status && j.status === 'error') throw new Error(j.message || 'update failed');
      } else {
        const r = await fetch(API.create, { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) });
        const j = await r.json();
        if(j.status && j.status === 'error') throw new Error(j.message || 'create failed');
      }

      modalRoot.style.display='none'; seasonModal.style.display='none';
      await fetchSeasons();
    } catch(e){
      alert('Save failed: ' + (e.message || 'error'));
      console.error('save error', e);
    } finally {
      modalSave.disabled = false;
    }
  });

  modalCancel.addEventListener('click', ()=> { modalRoot.style.display='none'; seasonModal.style.display='none'; });

  // delete season
  async function deleteSeason(id){
    try {
      const r = await fetch(API.remove, { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({id}) });
      const j = await r.json();
      if(j.status && j.status === 'error') throw new Error(j.message || 'delete failed');
      await fetchSeasons();
    } catch(e){
      alert('Delete failed: ' + (e.message || 'error'));
      console.error('deleteSeason', e);
    }
  }

  // open create / refresh wiring
  addBtn.addEventListener('click', async ()=>{
    await loadLanguages(); await loadHosts(); await loadContestants();
    openCreate();
  });
  refreshBtn.addEventListener('click', async ()=>{
    await loadLanguages(); await loadHosts(); await loadContestants(); await fetchSeasons();
  });

  searchInput.addEventListener('input', ()=>{
    const q = (searchInput.value || '').toLowerCase().trim();
    if(!q) return renderList(cachedSeasons || []);
    const filtered = (cachedSeasons || []).filter(s => {
      return (s.tag || '').toLowerCase().includes(q) || (String(s.season_no||'')).includes(q) || (String(s.year||'')).includes(q);
    });
    renderList(filtered);
  });

  filterLanguage.addEventListener('change', ()=>{
    const v = (filterLanguage.value || '').trim();
    if(!v) return renderList(cachedSeasons || []);
    renderList((cachedSeasons || []).filter(s => (s.language||'') === v));
  });

  filterShowType.addEventListener('change', ()=>{
    const v = (filterShowType.value || '').trim();
    if(!v) return renderList(cachedSeasons || []);
    renderList((cachedSeasons || []).filter(s => (s.show_type || '').toLowerCase() === v.toLowerCase()));
  });

  // init
  document.addEventListener('DOMContentLoaded', async ()=>{
    await loadLanguages(); await loadHosts(); await loadContestants(); await fetchSeasons();
  });

})();
