/* profiles.js
   - list profiles
   - add / edit / delete
   - preview / upload image (optional)
*/

(() => {
  const id = s => document.getElementById(s);
  const API = {
    list: '/dashboard/api/get_profiles.php',
    create: '/dashboard/api/save_profile.php',
    update: '/dashboard/api/update_profile.php',
    remove: '/dashboard/api/delete_profile.php',
    uploadImage: '/dashboard/api/upload_image.php'
  };

  const container = id('profilesContainer');
  const refreshBtn = id('refreshBtn');
  const addBtn = id('addBtn');
  const lastSync = id('lastSync');
  const globalSearch = id('globalSearch');

  // modal elements
  const modalRoot = id('modalRoot');
  const profileModal = id('profileModal');
  const modalTitle = id('modalTitle');
  const p_name = id('p_name');
  const p_title = id('p_title');
  const p_language = id('p_language');
  const p_bio = id('p_bio');
  const p_instagram = id('p_instagram');
  const p_twitter = id('p_twitter');
  const p_youtube = id('p_youtube');
  const p_photo = id('p_photo');
  const p_preview = id('p_preview');
  const modalCancel = id('modalCancel');
  const modalSave = id('modalSave');

  let editingId = null;
  let cachedList = [];

  function esc(s){ return (s||'').toString().replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }

  function renderList(items) {
    cachedList = items || [];
    container.innerHTML = '';
    if(!items || items.length === 0) {
      const empty = document.createElement('div');
      empty.className = 'recent-empty';
      empty.textContent = 'No profiles found. Click + Add Profile to create.';
      container.appendChild(empty);
      return;
    }

    items.forEach(item => {
      const row = document.createElement('div');
      row.className = 'recent-item';
      row.style.display = 'flex';
      row.style.alignItems = 'center';
      row.style.justifyContent = 'space-between';
      row.style.gap = '12px';

      const left = document.createElement('div');
      left.style.display = 'flex';
      left.style.alignItems = 'center';
      left.style.gap = '12px';

      const img = document.createElement('img');
      img.src = item.photo_url || 'images/avatar-placeholder.png';
      img.alt = item.name || '';
      img.style.width = '72px';
      img.style.height = '72px';
      img.style.borderRadius = '8px';
      img.style.objectFit = 'cover';
      img.style.border = '1px solid rgba(255,255,255,0.03)';

      const info = document.createElement('div');
      info.innerHTML = `<div style="font-weight:700">${esc(item.name || '')} <span style="font-weight:400;color:var(--muted)"> ${esc(item.title || '')}</span></div>
        <div style="font-size:13px;color:var(--muted)">${esc(item.language || '')}</div>
        <div style="margin-top:6px;color:var(--muted);font-size:13px;max-width:760px;">${esc(item.bio || '')}</div>`;

      left.appendChild(img);
      left.appendChild(info);

      const right = document.createElement('div');
      right.style.display = 'flex';
      right.style.flexDirection = 'column';
      right.style.alignItems = 'flex-end';
      right.style.gap = '8px';

      const actions = document.createElement('div');
      actions.style.display = 'flex';
      actions.style.gap = '8px';

      const viewBtn = document.createElement('button');
      viewBtn.className = 'btn small';
      viewBtn.textContent = 'View';
      viewBtn.addEventListener('click', () => openView(item));

      const editBtn = document.createElement('button');
      editBtn.className = 'btn small';
      editBtn.textContent = 'Edit';
      editBtn.addEventListener('click', () => openEdit(item));

      const delBtn = document.createElement('button');
      delBtn.className = 'btn small';
      delBtn.style.background = 'rgba(220,60,60,0.9)';
      delBtn.textContent = 'Delete';
      delBtn.addEventListener('click', ()=> {
        if(!confirm(`Delete profile "${item.name}"?`)) return;
        deleteProfile(item.id);
      });

      actions.appendChild(viewBtn);
      actions.appendChild(editBtn);
      actions.appendChild(delBtn);
      right.appendChild(actions);

      row.appendChild(left);
      row.appendChild(right);
      container.appendChild(row);
    });
  }

  async function fetchList() {
    container.innerHTML = '<div class="recent-empty">Loading profiles...</div>';
    try {
      const res = await fetch(API.list, {cache:'no-store'});
      if(!res.ok) throw new Error(`HTTP ${res.status}`);
      const json = await res.json();
      const rows = Array.isArray(json) ? json : (json.data || []);
      renderList(rows);
      lastSync.textContent = new Date().toLocaleString();
    } catch(e){
      container.innerHTML = `<div class="recent-empty">Unable to load profiles: ${e.message}</div>`;
      console.error('fetchList error', e);
    }
  }

  async function createProfile(payload) {
    const res = await fetch(API.create, {
      method: 'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify(payload)
    });
    const json = await res.json();
    if(json.status && json.status === 'error') throw new Error(json.message || 'Create failed');
    return json;
  }

  async function updateProfile(payload) {
    const res = await fetch(API.update, {
      method: 'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify(payload)
    });
    const json = await res.json();
    if(json.status && json.status === 'error') throw new Error(json.message || 'Update failed');
    return json;
  }

  async function deleteProfile(id) {
    const res = await fetch(API.remove, {
      method: 'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify({id})
    });
    const json = await res.json();
    if(json.status && json.status === 'error') throw new Error(json.message || 'Delete failed');
    await fetchList();
  }

  async function uploadImageFile(file) {
    if(!file) return null;
    const fd = new FormData();
    fd.append('image', file);
    const res = await fetch(API.uploadImage, { method:'POST', body: fd });
    const json = await res.json();
    if(json.status && json.status === 'error') throw new Error(json.message || 'Upload failed');
    // Expect json.url or similar
    return json.url || json.data?.url || json.path || null;
  }

  // modal handlers
  function clearModal() {
    editingId = null;
    p_name.value = '';
    p_title.value = '';
    p_language.value = '';
    p_bio.value = '';
    p_instagram.value = '';
    p_twitter.value = '';
    p_youtube.value = '';
    p_photo.value = '';
    p_preview.src = 'images/avatar-placeholder.png';
  }

  function showModal() {
    modalRoot.style.display = 'block';
    profileModal.style.display = 'block';
  }
  function hideModal() {
    modalRoot.style.display = 'none';
    profileModal.style.display = 'none';
    clearModal();
  }

  function openEdit(item) {
    editingId = item.id;
    modalTitle.textContent = 'Edit Profile';
    p_name.value = item.name || '';
    p_title.value = item.title || '';
    p_language.value = item.language || '';
    p_bio.value = item.bio || '';
    p_instagram.value = item.instagram || '';
    p_twitter.value = item.twitter || '';
    p_youtube.value = item.youtube || '';
    p_preview.src = item.photo_url || 'images/avatar-placeholder.png';
    showModal();
  }

  function openView(item) {
    // quick modal-like viewer using window.alert for simplicity or console
    const lines = [
      `Name: ${item.name || ''}`,
      `Title: ${item.title || ''}`,
      `Language: ${item.language || ''}`,
      `Bio: ${item.bio || ''}`,
      `Instagram: ${item.instagram || ''}`,
      `Twitter: ${item.twitter || ''}`,
      `YouTube: ${item.youtube || ''}`,
    ];
    alert(lines.join('\n'));
  }

  // preview image selection
  p_photo.addEventListener('change', (e) => {
    const f = e.target.files && e.target.files[0];
    if(!f) { p_preview.src = 'images/avatar-placeholder.png'; return; }
    const url = URL.createObjectURL(f);
    p_preview.src = url;
  });

  modalSave.addEventListener('click', async () => {
    const name = (p_name.value || '').trim();
    if(!name) return alert('Name required');
    modalSave.disabled = true;
    try {
      let photoUrl = null;
      if(p_photo.files && p_photo.files[0]) {
        try {
          photoUrl = await uploadImageFile(p_photo.files[0]);
        } catch(e) {
          if(!confirm('Image upload failed. Continue without image?')) {
            modalSave.disabled = false;
            return;
          }
        }
      }

      const payload = {
        name,
        title: p_title.value.trim(),
        language: p_language.value.trim(),
        bio: p_bio.value.trim(),
        instagram: p_instagram.value.trim(),
        twitter: p_twitter.value.trim(),
        youtube: p_youtube.value.trim()
      };
      if(photoUrl) payload.photo_url = photoUrl;

      if(editingId) {
        payload.id = editingId;
        await updateProfile(payload);
      } else {
        await createProfile(payload);
      }
      hideModal();
      await fetchList();
    } catch(e){
      alert('Save failed: ' + (e.message || 'error'));
      console.error('modalSave error', e);
    } finally {
      modalSave.disabled = false;
    }
  });

  modalCancel.addEventListener('click', hideModal);
  modalRoot.addEventListener('click', (ev) => { if(ev.target === modalRoot) hideModal(); });

  addBtn.addEventListener('click', () => {
    editingId = null;
    clearModal();
    modalTitle.textContent = 'Add Profile';
    showModal();
  });

  refreshBtn.addEventListener('click', fetchList);

  globalSearch.addEventListener('input', (e) => {
    const v = (e.target.value || '').toLowerCase().trim();
    if(!v) return renderList(cachedList || []);
    const filtered = (cachedList || []).filter(p => {
      return (p.name||'').toLowerCase().includes(v)
          || (p.title||'').toLowerCase().includes(v)
          || (p.language||'').toLowerCase().includes(v)
          || (p.bio||'').toLowerCase().includes(v);
    });
    renderList(filtered);
  });

  // initial load
  document.addEventListener('DOMContentLoaded', fetchList);
})();
