(() => {
  const id = s => document.getElementById(s);

  const API = {
    list: "/dashboard/api/languages/list.php",
    save: "/dashboard/api/languages/save.php",
    delete: "/dashboard/api/languages/delete.php"
  };

  const listArea = id("languagesList");
  const searchInput = id("searchInput");
  const addLanguageBtn = id("addLanguageBtn");

  const modal = id("langModal");
  const modalTitle = id("modalTitle");
  const langName = id("langName");
  const saveLangBtn = id("saveLangBtn");
  const closeModal = id("closeModal");

  let editingId = null;
  let cached = [];

  // Simple Escape
  const esc = s =>
    String(s || "").replace(/[&<>"']/g, c => ({
      "&": "&amp;",
      "<": "&lt;",
      ">": "&gt;",
      '"': "&quot;",
      "'": "&#39;"
    }[c]));

  // Modal controls
  function openModal(editId = null) {
    editingId = editId;
    langName.value = "";

    if (editId) {
      const found = cached.find(x => x.id == editId);
      if (found) langName.value = found.name;
      modalTitle.textContent = "Edit Language";
    } else {
      modalTitle.textContent = "Add Language";
    }

    modal.style.display = "flex";
  }

  closeModal.addEventListener("click", () => {
    modal.style.display = "none";
  });

  addLanguageBtn.addEventListener("click", () => openModal());

  // Save language
  saveLangBtn.addEventListener("click", async () => {
    const name = langName.value.trim();
    if (!name) return alert("Enter language name");

    try {
      const res = await fetch(API.save, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          id: editingId,
          name: name
        })
      });

      const j = await res.json();
      if (j.status !== "success") throw new Error(j.message);

      modal.style.display = "none";
      loadLanguages();
    } catch (e) {
      alert("Error saving: " + e.message);
    }
  });

  // Delete
  async function deleteLang(id) {
    if (!confirm("Delete this language?")) return;

    try {
      const res = await fetch(API.delete, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ id })
      });

      const j = await res.json();
      if (j.status !== "success") throw new Error(j.message);

      loadLanguages();
    } catch (e) {
      alert("Delete failed: " + e.message);
    }
  }

  // Render list
  function render(list) {
    listArea.innerHTML = "";

    if (!list.length) {
      listArea.innerHTML = `<div class="empty">No languages found</div>`;
      return;
    }

    list.forEach(l => {
      const row = document.createElement("div");
      row.className = "list-item";

      row.innerHTML = `
        <div class="left">
          <div class="title">${esc(l.name)}</div>
          <div class="meta">ID: ${l.id}</div>
        </div>

        <div class="right">
          <button class="btn small" data-edit="${l.id}">Edit</button>
          <button class="btn small danger" data-del="${l.id}">Delete</button>
        </div>
      `;

      // Edit / Delete clicks
      row.querySelector("[data-edit]").onclick = () => openModal(l.id);
      row.querySelector("[data-del]").onclick = () => deleteLang(l.id);

      listArea.appendChild(row);
    });
  }

  // Load languages from API
  async function loadLanguages() {
    listArea.innerHTML = `<div class="loading">Loading…</div>`;
    try {
      const res = await fetch(API.list);
      const j = await res.json();
      cached = j.data || j;  // support old/new structure
      render(cached);
    } catch (e) {
      listArea.innerHTML = `<div class="error">Failed to load</div>`;
    }
  }

  // Search filter
  searchInput.addEventListener("input", () => {
    const q = searchInput.value.toLowerCase();
    const filtered = cached.filter(x => x.name.toLowerCase().includes(q));
    render(filtered);
  });

  // Init
  loadLanguages();
})();
