/* hosts.js
   - List / Add / Edit / Delete hosts
   - Working years field supports formats:
     - "2017-2024"
     - "2018-"
     - "2015"
   - photo preview + optional upload to /dashboard/api/upload_image.php
*/

(() => {
  const id = s => document.getElementById(s);
  const API = {
    list: '/dashboard/api/get_hosts.php',
    create: '/dashboard/api/save_host.php',
    update: '/dashboard/api/update_host.php',
    remove: '/dashboard/api/delete_host.php',
    uploadImage: '/dashboard/api/upload_image.php'
  };

  // elements
  const container = id('hostsContainer');
  const refreshBtn = id('refreshBtn');
  const addBtn = id('addBtn');
  const lastSync = id('lastSync');
  const globalSearch = id('globalSearch');

  // modal
  const modalRoot = id('modalRoot');
  const hostModal = id('hostModal');
  const modalTitle = id('modalTitle');
  const h_name = id('h_name');
  const h_years = id('h_years');
  const h_bio = id('h_bio');
  const h_instagram = id('h_instagram');
  const h_photo = id('h_photo');
  const h_preview = id('h_preview');
  const modalCancel = id('modalCancel');
  const modalSave = id('modalSave');

  let cached = [];
  let editingId = null;

  function esc(s){ return (s||'').toString().replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }

  function formatYearsLabel(years){
    if(!years) return '';
    // Accept "2017-2024", "2018-", "2015"
    const cleaned = String(years).trim();
    if(/\d{4}\s*-\s*\d{4}/.test(cleaned)) return cleaned.replace(/\s+/g,'');
    if(/\d{4}\s*-\s*$/.test(cleaned) || /\d{4}\s*-\s*$/.test(cleaned)) return cleaned.replace(/\s+/g,'');
    if(/^\d{4}$/.test(cleaned)) return cleaned;
    return cleaned;
  }

  function renderList(list){
    cached = list || [];
    container.innerHTML = '';
    if(!cached.length){
      container.innerHTML = '<div class="recent-empty">No hosts found. Click + Add Host to create.</div>';
      return;
    }

    cached.forEach(h => {
      const row = document.createElement('div');
      row.className = 'recent-item';
      row.style.display='flex';
      row.style.alignItems='center';
      row.style.justifyContent='space-between';
      row.style.gap='12px';

      const left = document.createElement('div');
      left.style.display='flex'; left.style.alignItems='center'; left.style.gap='12px';

      const img = document.createElement('img');
      img.src = h.photo || 'images/avatar-placeholder.png';
      img.style.width='72px'; img.style.height='72px'; img.style.borderRadius='8px'; img.style.objectFit='cover'; img.style.border='1px solid rgba(255,255,255,0.03)';

      const info = document.createElement('div');
      const years = formatYearsLabel(h.years || '');
      info.innerHTML = `<div style="font-weight:700">${esc(h.name || '')} <span style="font-weight:400;color:var(--muted);font-size:13px;">${years?('('+esc(years)+')') : ''}</span></div>
                        <div style="font-size:13px;color:var(--muted);margin-top:6px;">${esc((h.bio||'').slice(0,120))}</div>
                        <div style="font-size:12px;color:var(--muted);margin-top:6px;">${h.instagram?'<a href="'+esc(h.instagram)+'" target="_blank">Instagram</a>':''}</div>`;

      left.appendChild(img);
      left.appendChild(info);

      const right = document.createElement('div');
      right.style.display='flex'; right.style.flexDirection='column'; right.style.alignItems='flex-end'; right.style.gap='8px';
      const actions = document.createElement('div'); actions.style.display='flex'; actions.style.gap='8px';

      const editBtn = document.createElement('button'); editBtn.className='btn small'; editBtn.textContent='Edit';
      const delBtn = document.createElement('button'); delBtn.className='btn small'; delBtn.style.background='rgba(220,60,60,0.9)'; delBtn.textContent='Delete';
      editBtn.addEventListener('click', ()=> openEdit(h));
      delBtn.addEventListener('click', ()=> { if(!confirm('Delete host?')) return; deleteHost(h.id || h.uid || h.name); });

      actions.appendChild(editBtn); actions.appendChild(delBtn);
      right.appendChild(actions);
      row.appendChild(left); row.appendChild(right);

      container.appendChild(row);
    });
  }

  async function fetchHosts(){
    container.innerHTML = '<div class="recent-empty">Loading hosts...</div>';
    try {
      const r = await fetch(API.list, {cache:'no-store'});
      if(!r.ok) throw new Error('HTTP ' + r.status);
      const json = await r.json();
      const rows = Array.isArray(json) ? json : (json.data || []);
      renderList(rows);
      lastSync.textContent = new Date().toLocaleString();
    } catch(e){
      container.innerHTML = `<div class="recent-empty">Failed to load hosts: ${e.message}</div>`;
      console.error('fetchHosts', e);
    }
  }

  function openAdd(){
    editingId = null;
    modalTitle.textContent = 'Add Host';
    h_name.value = ''; h_years.value = ''; h_bio.value = ''; h_instagram.value = ''; h_photo.value = ''; h_preview.src = 'images/avatar-placeholder.png';
    modalRoot.style.display='block'; hostModal.style.display='block';
  }

  function openEdit(h){
    editingId = h.id || h.uid || h.name;
    modalTitle.textContent = 'Edit Host';
    h_name.value = h.name || '';
    h_years.value = h.years || '';
    h_bio.value = h.bio || '';
    h_instagram.value = h.instagram || '';
    h_preview.src = h.photo || 'images/avatar-placeholder.png';
    modalRoot.style.display='block'; hostModal.style.display='block';
  }

  // preview image
  h_photo.addEventListener('change', (e) => {
    const f = e.target.files && e.target.files[0];
    if(!f){ h_preview.src = 'images/avatar-placeholder.png'; return; }
    h_preview.src = URL.createObjectURL(f);
  });

  // save
  modalSave.addEventListener('click', async () => {
    const name = (h_name.value || '').trim();
    if(!name) return alert('Enter host name');
    modalSave.disabled = true;
    try {
      let photoUrl = null;
      if(h_photo.files && h_photo.files[0]){
        try {
          const fd = new FormData(); fd.append('image', h_photo.files[0]);
          const up = await fetch(API.uploadImage, { method:'POST', body: fd });
          const j = await up.json();
          if(j.status && j.status === 'error') throw new Error(j.message || 'upload failed');
          photoUrl = j.url || j.data?.url || j.path || null;
        } catch(e){
          if(!confirm('Photo upload failed. Continue without photo?')) { modalSave.disabled = false; return; }
        }
      }

      const payload = {
        name,
        years: h_years.value.trim(),
        bio: h_bio.value.trim(),
        instagram: h_instagram.value.trim()
      };
      if(photoUrl) payload.photo = photoUrl;
      if(editingId) payload.id = editingId;

      const url = editingId ? API.update : API.create;
      const r = await fetch(url, { method: 'POST', headers: {'Content-Type':'application/json'}, body: JSON.stringify(payload) });
      const j = await r.json();
      if(j.status && j.status === 'error') throw new Error(j.message || 'save failed');

      modalRoot.style.display='none'; hostModal.style.display='none';
      await fetchHosts();
    } catch(e){
      alert('Save failed: ' + (e.message || 'error'));
      console.error('save', e);
    } finally { modalSave.disabled = false; }
  });

  modalCancel.addEventListener('click', ()=> { modalRoot.style.display='none'; hostModal.style.display='none'; });

  async function deleteHost(id){
    try {
      const r = await fetch(API.remove, { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ id }) });
      const j = await r.json();
      if(j.status && j.status === 'error') throw new Error(j.message || 'delete failed');
      await fetchHosts();
    } catch(e){
      alert('Delete failed: ' + (e.message || 'error'));
      console.error('deleteHost', e);
    }
  }

  // search
  globalSearch.addEventListener('input', ()=>{
    const q = (globalSearch.value || '').toLowerCase().trim();
    if(!q) return renderList(cached);
    const filtered = (cached || []).filter(h => {
      return (h.name || '').toLowerCase().includes(q) ||
             (h.years || '').toLowerCase().includes(q) ||
             (h.bio || '').toLowerCase().includes(q);
    });
    renderList(filtered);
  });

  // init
  addBtn.addEventListener('click', openAdd);
  refreshBtn.addEventListener('click', fetchHosts);
  document.addEventListener('DOMContentLoaded', fetchHosts);

})();
