/* dashboard_home.js
   drop in /dashboard/js/
   - Fetches counts from your API endpoints (graceful if an endpoint missing)
   - Handles theme toggle (persisted in localStorage)
   - Small "Refresh" to re-fetch counts
*/

(() => {
  const el = id => document.getElementById(id);
  const endpoints = {
    languages: '/dashboard/api/get_languages.php',
    hosts: '/dashboard/api/get_hosts.php',
    contestants: '/dashboard/api/get_contestants.php',
    seasons: '/dashboard/api/get_seasons.php',
    broadcasts: '/dashboard/api/get_broadcasts.php'
  };

  // helper to fetch list and return length; endpoint may return JSON array or object with "data" or something.
  async function fetchCount(url){
    try {
      const r = await fetch(url, {cache:'no-store'});
      if(!r.ok) throw new Error(`HTTP ${r.status}`);
      const json = await r.json();
      // If API returns object like {status:'success', data: [...]}
      if(Array.isArray(json)) return json.length;
      if(json && Array.isArray(json.data)) return json.data.length;
      // If API returns rows count directly like {count: 5}
      if(json && typeof json.count === 'number') return json.count;
      // If it's object keyed by rows, try to count top-level keys
      if(json && typeof json === 'object') return Object.keys(json).length;
      return 0;
    } catch (err){
      // endpoint missing or parse error -> return 0 & console.log
      console.warn('fetchCount error', url, err.message);
      return 0;
    }
  }

  async function refreshAll(){
    el('refreshBtn').disabled = true;
    el('refreshBtn').textContent = 'Refreshing...';

    const [langs, hosts, conts, seasons, broadcasts] = await Promise.all([
      fetchCount(endpoints.languages),
      fetchCount(endpoints.hosts),
      fetchCount(endpoints.contestants),
      fetchCount(endpoints.seasons),
      fetchCount(endpoints.broadcasts)
    ]);

    el('countLanguages').textContent = langs;
    el('countHosts').textContent = hosts;
    el('countContestants').textContent = conts;
    el('countSeasons').textContent = seasons;
    el('countBroadcasts').textContent = broadcasts;

    el('lastSync').textContent = new Date().toLocaleString();
    el('refreshBtn').disabled = false;
    el('refreshBtn').textContent = 'Refresh';
    buildRecentList();
  }

  // small recent updates example (non-destructive). Ideally your API exposes recent items — fallback uses counts change.
  async function buildRecentList(){
    const container = el('recentList');
    container.innerHTML = ''; // clear

    // Try to fetch latest seasons first (if endpoint supports returning array)
    try{
      const resp = await fetch('/dashboard/api/get_seasons.php');
      if (resp.ok) {
        const json = await resp.json();
        const rows = Array.isArray(json) ? json.slice(0,6) : (json.data && Array.isArray(json.data) ? json.data.slice(0,6) : []);
        if (rows && rows.length){
          rows.forEach(r=>{
            const node = document.createElement('div');
            node.className = 'recent-item';
            node.innerHTML = `<div><strong>${r.name || r.title || r.season || 'Season'}</strong><div style="color:var(--muted);font-size:12px">${r.year || r.created_at || ''}</div></div>
                              <div style="color:var(--muted);font-size:12px">${r.participants ? (r.participants + ' participants') : ''}</div>`;
            container.appendChild(node);
          });
          return;
        }
      }
    }catch(e){
      // ignore
    }

    // fallback: simple placeholder
    const placeholder = document.createElement('div');
    placeholder.className = 'recent-empty';
    placeholder.textContent = 'No recent items available — try Refresh.';
    container.appendChild(placeholder);
  }

  // theme handling
  function applyTheme(dark){
    if(dark){
      document.documentElement.classList.remove('light');
      el('themeToggle').checked = true;
      el('themeLabel').textContent = 'Dark';
    } else {
      document.documentElement.classList.add('light');
      el('themeToggle').checked = false;
      el('themeLabel').textContent = 'Light';
    }
  }

  function initTheme(){
    const saved = localStorage.getItem('bb_theme');
    const isDark = saved === null ? true : (saved === 'dark');
    applyTheme(isDark);
    el('themeToggle').addEventListener('change', (e)=>{
      const dark = e.target.checked;
      applyTheme(dark);
      localStorage.setItem('bb_theme', dark ? 'dark' : 'light');
    });
  }

  // sidebar collapse small behaviour
  function initSidebar(){
    const sidebar = document.getElementById('sidebar');
    const btn = document.getElementById('sidebarCollapse');
    btn.addEventListener('click', ()=>{
      sidebar.classList.toggle('collapsed');
    });
  }

  // wire up
  document.addEventListener('DOMContentLoaded', ()=>{
    initTheme();
    initSidebar();

    el('refreshBtn').addEventListener('click', refreshAll);
    refreshAll();
  });
})();
