/* contestants.js
   - list contestants
   - add / edit / delete
   - import CSV
   - preview image and optional upload
   - works with /dashboard/api/* endpoints
*/

(() => {
  const id = s => document.getElementById(s);
  const API = {
    list: '/dashboard/api/get_contestants.php',
    create: '/dashboard/api/save_contestant.php',
    update: '/dashboard/api/update_contestant.php',
    remove: '/dashboard/api/delete_contestant.php',
    uploadImage: '/dashboard/api/upload_image.php',
    seasons: '/dashboard/api/get_seasons.php',
    languages: '/dashboard/api/get_languages.php'
  };

  // elements
  const container = id('contestantsContainer');
  const refreshBtn = id('refreshBtn');
  const addBtn = id('addBtn');
  const importBtn = id('importBtn');
  const importModal = id('importModal');
  const importCancel = id('importCancel');
  const importStart = id('importStart');
  const csvFile = id('csvFile');
  const lastSync = id('lastSync');

  const filterLanguage = id('filterLanguage');
  const filterSeason = id('filterSeason');
  const filterShowType = id('filterShowType');
  const globalSearch = id('globalSearch');

  // modal elements
  const modalRoot = id('modalRoot');
  const contestantModal = id('contestantModal');
  const modalTitle = id('modalTitle');
  const c_name = id('c_name');
  const c_language = id('c_language');
  const c_show_type = id('c_show_type');
  const c_season_no = id('c_season_no');
  const c_bio = id('c_bio');
  const c_photo = id('c_photo');
  const c_preview = id('c_preview');
  const modalCancel = id('modalCancel');
  const modalSave = id('modalSave');

  let cachedList = [];
  let seasonsList = [];
  let languagesList = [];
  let editingId = null;

  // helpers
  function esc(s){ return (s||'').toString().replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }
  function showModal(modal){ modalRoot.style.display = 'block'; modal.style.display = 'block'; }
  function hideModal(modal){ modalRoot.style.display = 'none'; modal.style.display = 'none'; }

  // fetch languages & seasons
  async function loadLanguages() {
    try {
      const r = await fetch(API.languages, {cache:'no-store'});
      const json = await r.json();
      const arr = Array.isArray(json) ? json : (json.data || []);
      languagesList = arr;
      c_language.innerHTML = '<option value="">(choose)</option>' + arr.map(x=>`<option value="${x.id}">${esc(x.name)}</option>`).join('');
      filterLanguage.innerHTML = '<option value="">All languages</option>' + arr.map(x=>`<option value="${esc(x.name)}">${esc(x.name)}</option>`).join('');
    } catch(e){
      console.warn('loadLanguages', e);
      c_language.innerHTML = '<option value="">(no languages)</option>';
    }
  }

  async function loadSeasons(){
    try {
      const r = await fetch(API.seasons, {cache:'no-store'});
      const json = await r.json();
      const arr = Array.isArray(json) ? json : (json.data || []);
      seasonsList = arr;
      filterSeason.innerHTML = '<option value="">All seasons</option>' + arr.map(s=>`<option value="${esc(s.season_no || s.id)}">${esc(s.tag || s.season_no || s.id)}</option>`).join('');
    } catch(e){
      console.warn('loadSeasons', e);
      filterSeason.innerHTML = '<option value="">All seasons</option>';
    }
  }

  // render list
  function renderList(items){
    cachedList = items;
    container.innerHTML = '';
    if(!items || items.length === 0){
      const empty = document.createElement('div');
      empty.className = 'recent-empty';
      empty.textContent = 'No contestants found. Click + Add Contestant to create.';
      container.appendChild(empty);
      return;
    }

    items.forEach(item => {
      const row = document.createElement('div');
      row.className = 'recent-item';
      row.style.display = 'flex';
      row.style.alignItems = 'center';
      row.style.justifyContent = 'space-between';
      row.style.gap = '12px';

      const left = document.createElement('div');
      left.style.display = 'flex';
      left.style.alignItems = 'center';
      left.style.gap = '12px';

      const img = document.createElement('img');
      img.src = item.photo_url || 'images/avatar-placeholder.png';
      img.alt = item.name || '';
      img.style.width = '72px';
      img.style.height = '72px';
      img.style.borderRadius = '8px';
      img.style.objectFit = 'cover';
      img.style.border = '1px solid rgba(255,255,255,0.03)';

      const info = document.createElement('div');
      info.innerHTML = `<div style="font-weight:700">${esc(item.name || '')}</div>
        <div style="font-size:13px;color:var(--muted)">${esc(item.language || '')} • ${esc(item.show_type || '')} ${item.season_no ? ('• S' + esc(item.season_no)) : ''}</div>
        <div style="font-size:12px;color:var(--muted); margin-top:6px; max-width:640px;">${esc(item.bio || '')}</div>`;

      left.appendChild(img);
      left.appendChild(info);

      const right = document.createElement('div');
      right.style.display = 'flex';
      right.style.flexDirection = 'column';
      right.style.alignItems = 'flex-end';
      right.style.gap = '8px';

      const btns = document.createElement('div');
      btns.style.display = 'flex';
      btns.style.gap = '8px';

      const editBtn = document.createElement('button');
      editBtn.className = 'btn small';
      editBtn.textContent = 'Edit';
      editBtn.addEventListener('click', ()=> openEdit(item));

      const delBtn = document.createElement('button');
      delBtn.className = 'btn small';
      delBtn.style.background = 'rgba(220,60,60,0.9)';
      delBtn.textContent = 'Delete';
      delBtn.addEventListener('click', ()=> {
        if(!confirm(`Delete contestant "${item.name}"?`)) return;
        deleteContestant(item.id);
      });

      btns.appendChild(editBtn);
      btns.appendChild(delBtn);

      right.appendChild(btns);
      row.appendChild(left);
      row.appendChild(right);
      container.appendChild(row);
    });
  }

  // fetch list
  async function fetchList(){
    container.innerHTML = '<div class="recent-empty">Loading contestants...</div>';
    try {
      const r = await fetch(API.list, {cache:'no-store'});
      if(!r.ok) throw new Error(`HTTP ${r.status}`);
      const json = await r.json();
      const rows = Array.isArray(json) ? json : (json.data || []);
      renderList(rows);
      lastSync.textContent = new Date().toLocaleString();
    } catch(e){
      container.innerHTML = `<div class="recent-empty">Unable to load contestants: ${e.message}</div>`;
      console.error('fetchList error', e);
    }
  }

  // create/update/delete
  async function createContestant(payload){
    const r = await fetch(API.create, {
      method:'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify(payload)
    });
    const json = await r.json();
    if(json.status && json.status === 'error') throw new Error(json.message || 'create failed');
    return json;
  }
  async function updateContestant(payload){
    const r = await fetch(API.update, {
      method:'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify(payload)
    });
    const json = await r.json();
    if(json.status && json.status === 'error') throw new Error(json.message || 'update failed');
    return json;
  }
  async function deleteContestant(id){
    const r = await fetch(API.remove, {
      method:'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify({id})
    });
    const json = await r.json();
    if(json.status && json.status === 'error') throw new Error(json.message || 'delete failed');
    await fetchList();
  }

  // upload image helper (optional)
  async function uploadImageFile(file){
    if(!file) return null;
    const fd = new FormData();
    fd.append('image', file);
    const r = await fetch(API.uploadImage, { method:'POST', body: fd });
    if(!r.ok) throw new Error('upload failed');
    const json = await r.json();
    if(json.status === 'error') throw new Error(json.message || 'upload error');
    // try to return url
    return json.url || json.data?.url || json.path || null;
  }

  // modal handling
  function openAdd(){
    editingId = null;
    modalTitle.textContent = 'Add Contestant';
    c_name.value = '';
    c_language.value = '';
    c_show_type.value = '';
    c_season_no.value = '';
    c_bio.value = '';
    c_photo.value = '';
    c_preview.src = 'images/avatar-placeholder.png';
    showModal(contestantModal);
  }

  function openEdit(item){
    editingId = item.id;
    modalTitle.textContent = 'Edit Contestant';
    c_name.value = item.name || '';
    c_language.value = item.language || '';
    c_show_type.value = item.show_type || '';
    c_season_no.value = item.season_no || '';
    c_bio.value = item.bio || '';
    c_preview.src = item.photo_url || 'images/avatar-placeholder.png';
    showModal(contestantModal);
  }

  // preview image
  c_photo.addEventListener('change', (e) => {
    const f = e.target.files && e.target.files[0];
    if(!f) { c_preview.src = 'images/avatar-placeholder.png'; return; }
    c_preview.src = URL.createObjectURL(f);
  });

  // save modal
  modalSave.addEventListener('click', async () => {
    const name = (c_name.value || '').trim();
    if(!name) return alert('Name required');
    modalSave.disabled = true;
    try {
      let photoUrl = null;
      if(c_photo.files && c_photo.files[0]) {
        try {
          photoUrl = await uploadImageFile(c_photo.files[0]);
        } catch(e){
          if(!confirm('Image upload failed. Continue without uploading image?')) {
            modalSave.disabled = false;
            return;
          }
        }
      }

      const payload = {
        name,
        language: c_language.value || '',
        show_type: c_show_type.value || '',
        season_no: c_season_no.value || '',
        bio: c_bio.value || ''
      };
      if(photoUrl) payload.photo_url = photoUrl;
      if(editingId) {
        payload.id = editingId;
        await updateContestant(payload);
      } else {
        await createContestant(payload);
      }
      hideModal(contestantModal);
      await fetchList();
    } catch(e){
      alert('Save failed: ' + (e.message || 'error'));
      console.error('save error', e);
    } finally {
      modalSave.disabled = false;
    }
  });

  modalCancel.addEventListener('click', () => hideModal(contestantModal));
  // close import modal
  importCancel.addEventListener('click', () => hideModal(importModal));

  // import CSV
  importBtn.addEventListener('click', () => {
    csvFile.value = '';
    showModal(importModal);
  });

  importStart.addEventListener('click', async () => {
    const f = csvFile.files && csvFile.files[0];
    if(!f) return alert('Choose a CSV file first');
    importStart.disabled = true;
    try {
      const text = await f.text();
      // parse simple CSV (no quotes handling). Expect header row.
      const lines = text.split(/\r?\n/).filter(Boolean);
      if(lines.length < 2) throw new Error('CSV has no rows');
      const headers = lines[0].split(',').map(h=>h.trim().toLowerCase());
      const rows = lines.slice(1).map(l => l.split(',').map(c=>c.trim()));
      // for each row, build payload
      for(const r of rows){
        const obj = {};
        headers.forEach((h,i) => { obj[h] = (r[i] || '').trim(); });
        // map fields
        const payload = {
          name: obj.name || '',
          language: obj.language || '',
          show_type: obj.show_type || '',
          season_no: obj.season_no || '',
          bio: obj.bio || '',
          photo_url: obj.photo_url || ''
        };
        if(!payload.name) continue;
        await createContestant(payload);
      }
      hideModal(importModal);
      await fetchList();
      alert('Import finished');
    } catch(e){
      alert('Import failed: ' + (e.message || 'error'));
      console.error('import error', e);
    } finally {
      importStart.disabled = false;
    }
  });

  // filters
  filterLanguage.addEventListener('change', applyFilters);
  filterSeason.addEventListener('change', applyFilters);
  filterShowType.addEventListener('change', applyFilters);

  globalSearch.addEventListener('input', () => {
    applyFilters();
  });

  function applyFilters(){
    const lang = (filterLanguage.value || '').toLowerCase();
    const season = (filterSeason.value || '').toLowerCase();
    const showType = (filterShowType.value || '').toLowerCase();
    const q = (globalSearch.value || '').toLowerCase().trim();

    const filtered = (cachedList || []).filter(item => {
      if(lang && ((item.language||'').toLowerCase() !== lang)) return false;
      if(season && String(item.season_no || item.season_id || '').toLowerCase() !== season) return false;
      if(showType && ((item.show_type||'').toLowerCase() !== showType)) return false;
      if(q) {
        const hay = ((item.name||'') + ' ' + (item.bio||'') + ' ' + (item.language||'') + ' ' + (item.show_type||'')).toLowerCase();
        if(!hay.includes(q)) return false;
      }
      return true;
    });

    renderList(filtered);
  }

  // initial load
  refreshBtn.addEventListener('click', async () => {
    await loadLanguages();
    await loadSeasons();
    await fetchList();
  });

  addBtn.addEventListener('click', openAdd);

  document.addEventListener('DOMContentLoaded', async () => {
    await loadLanguages();
    await loadSeasons();
    await fetchList();
  });

})();
