/* broadcasts.js
   - list broadcasts (OTT / Satellite / TV)
   - add / edit / delete
   - preview and optional upload of logo (upload_image.php)
   - flexible parsing of API responses
*/

(() => {
  const id = s => document.getElementById(s);
  const API = {
    list: '/dashboard/api/get_broadcasts.php',
    create: '/dashboard/api/save_broadcast.php',
    update: '/dashboard/api/update_broadcast.php',
    remove: '/dashboard/api/delete_broadcast.php',
    uploadImage: '/dashboard/api/upload_image.php',
    languages: '/dashboard/api/get_languages.php'
  };

  // elems
  const container = id('broadcastsContainer');
  const refreshBtn = id('refreshBtn');
  const addBtn = id('addBtn');
  const lastSync = id('lastSync');
  const filterLanguage = id('filterLanguage');
  const filterType = id('filterType');
  const globalSearch = id('globalSearch');

  // modal elements
  const modalRoot = id('modalRoot');
  const broadcastModal = id('broadcastModal');
  const modalTitle = id('modalTitle');
  const b_type = id('b_type');
  const b_language = id('b_language');
  const b_name = id('b_name');
  const b_link = id('b_link');
  const b_logo_input = id('b_logo_input');
  const b_logo_preview = id('b_logo_preview');
  const b_notes = id('b_notes');
  const modalCancel = id('modalCancel');
  const modalSave = id('modalSave');

  let cachedList = [];
  let editingId = null;

  function esc(s){ return (s||'').toString().replace(/[&<>"']/g, c => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[c])); }

  async function loadLanguages(){
    try {
      const r = await fetch(API.languages, {cache:'no-store'});
      const json = await r.json();
      const arr = Array.isArray(json) ? json : (json.data || []);
      b_language.innerHTML = '<option value="">(choose)</option>' + arr.map(l => `<option value="${esc(l.name)}">${esc(l.name)}</option>`).join('');
      filterLanguage.innerHTML = '<option value="">All languages</option>' + arr.map(l => `<option value="${esc(l.name)}">${esc(l.name)}</option>`).join('');
    } catch(e){
      b_language.innerHTML = '<option value="">(no languages)</option>';
    }
  }

  function renderList(items){
    cachedList = items || [];
    container.innerHTML = '';
    if(!items || items.length === 0){
      container.innerHTML = '<div class="recent-empty">No broadcasts found. Click + Add Broadcast to create one.</div>';
      return;
    }

    items.forEach(it => {
      const row = document.createElement('div');
      row.className = 'recent-item';
      row.style.display = 'flex';
      row.style.justifyContent = 'space-between';
      row.style.alignItems = 'center';
      row.style.gap = '12px';

      const left = document.createElement('div'); left.style.display='flex'; left.style.gap='12px'; left.style.alignItems='center';
      const img = document.createElement('img');
      img.src = it.logo || 'images/avatar-placeholder.png'; img.style.width='72px'; img.style.height='40px'; img.style.objectFit='contain'; img.style.borderRadius='6px'; img.style.border='1px solid rgba(255,255,255,0.04)';
      const info = document.createElement('div'); info.innerHTML = `<div style="font-weight:700">${esc(it.name || '')}</div><div style="font-size:13px;color:var(--muted)">${esc(it.type || it.broadcast_type || '')} • ${esc(it.language || '')}</div><div style="margin-top:6px;color:var(--muted);font-size:13px;">${esc(it.notes || '')}</div>`;
      left.appendChild(img); left.appendChild(info);

      const right = document.createElement('div'); right.style.display='flex'; right.style.gap='8px';
      const editBtn = document.createElement('button'); editBtn.className='btn small'; editBtn.textContent='Edit';
      const delBtn = document.createElement('button'); delBtn.className='btn small'; delBtn.style.background='rgba(220,60,60,0.9)'; delBtn.textContent='Delete';
      const openBtn = document.createElement('a'); openBtn.className='btn small'; openBtn.textContent='Open'; openBtn.href = it.link || '#'; openBtn.target='_blank';
      editBtn.addEventListener('click', ()=> openEdit(it));
      delBtn.addEventListener('click', ()=> { if(!confirm(`Delete "${it.name}"?`)) return; deleteBroadcast(it.id); });
      right.appendChild(openBtn); right.appendChild(editBtn); right.appendChild(delBtn);

      row.appendChild(left); row.appendChild(right);
      container.appendChild(row);
    });
  }

  async function fetchList(){
    container.innerHTML = '<div class="recent-empty">Loading broadcasts...</div>';
    try {
      const r = await fetch(API.list, {cache:'no-store'});
      if(!r.ok) throw new Error(`HTTP ${r.status}`);
      const json = await r.json();
      const rows = Array.isArray(json) ? json : (json.data || []);
      renderList(rows);
      lastSync.textContent = new Date().toLocaleString();
    } catch(e){
      container.innerHTML = `<div class="recent-empty">Failed to load: ${e.message}</div>`;
      console.error('fetchList', e);
    }
  }

  // upload helper
  async function uploadImageFile(file){
    if(!file) return null;
    try {
      const fd = new FormData(); fd.append('image', file);
      const r = await fetch(API.uploadImage, { method:'POST', body: fd });
      if(!r.ok) throw new Error('Upload HTTP ' + r.status);
      const json = await r.json();
      if(json.status && json.status === 'error') throw new Error(json.message || 'upload failed');
      return json.url || json.data?.url || json.path || null;
    } catch(e){
      console.error('uploadImageFile', e);
      throw e;
    }
  }

  // modal open / edit
  function openAdd(){
    editingId = null;
    modalTitle.textContent = 'Add Broadcast';
    b_type.value = 'ott'; b_language.value = ''; b_name.value=''; b_link.value=''; b_notes.value=''; b_logo_input.value=''; b_logo_preview.src='images/avatar-placeholder.png';
    modalRoot.style.display='block'; broadcastModal.style.display='block';
  }
  function openEdit(item){
    editingId = item.id;
    modalTitle.textContent = 'Edit Broadcast';
    b_type.value = item.type || item.broadcast_type || 'ott';
    b_language.value = item.language || '';
    b_name.value = item.name || '';
    b_link.value = item.link || item.url || '';
    b_notes.value = item.notes || '';
    b_logo_preview.src = item.logo || 'images/avatar-placeholder.png';
    modalRoot.style.display='block'; broadcastModal.style.display='block';
  }

  // delete
  async function deleteBroadcast(id){
    try {
      const r = await fetch(API.remove, { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({id}) });
      const j = await r.json();
      if(j.status && j.status === 'error') throw new Error(j.message || 'delete failed');
      await fetchList();
    } catch(e){
      alert('Delete failed: ' + (e.message || 'error'));
      console.error(e);
    }
  }

  // events
  b_logo_input.addEventListener('change', (e)=> {
    const f = e.target.files && e.target.files[0];
    if(!f) { b_logo_preview.src = 'images/avatar-placeholder.png'; return; }
    b_logo_preview.src = URL.createObjectURL(f);
  });

  modalSave.addEventListener('click', async ()=>{
    const name = (b_name.value || '').trim();
    if(!name) return alert('Name required');
    modalSave.disabled = true;
    try {
      let logoUrl = null;
      if(b_logo_input.files && b_logo_input.files[0]) {
        try { logoUrl = await uploadImageFile(b_logo_input.files[0]); } catch(e){ if(!confirm('Logo upload failed. Continue without logo?')) { modalSave.disabled=false; return; } }
      }
      const payload = {
        type: b_type.value || '',
        language: b_language.value || '',
        name,
        link: b_link.value.trim(),
        notes: b_notes.value.trim()
      };
      if(logoUrl) payload.logo = logoUrl;

      if(editingId){
        payload.id = editingId;
        const r = await fetch(API.update, { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) });
        const j = await r.json();
        if(j.status && j.status === 'error') throw new Error(j.message || 'update failed');
      } else {
        const r = await fetch(API.create, { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) });
        const j = await r.json();
        if(j.status && j.status === 'error') throw new Error(j.message || 'create failed');
      }
      modalRoot.style.display='none'; broadcastModal.style.display='none';
      await fetchList();
    } catch(e){
      alert('Save failed: ' + (e.message || 'error'));
      console.error('save', e);
    } finally {
      modalSave.disabled = false;
    }
  });

  modalCancel.addEventListener('click', ()=> { modalRoot.style.display='none'; broadcastModal.style.display='none'; });

  // filtering & search
  filterLanguage.addEventListener('change', applyFilter);
  filterType.addEventListener('change', applyFilter);
  globalSearch.addEventListener('input', applyFilter);

  function applyFilter(){
    const lang = (filterLanguage.value || '').toLowerCase();
    const type = (filterType.value || '').toLowerCase();
    const q = (globalSearch.value || '').toLowerCase().trim();
    const filtered = (cachedList || []).filter(it => {
      if(lang && ((it.language||'').toLowerCase() !== lang)) return false;
      if(type && ((it.type||it.broadcast_type||'').toLowerCase() !== type)) return false;
      if(q) {
        const hay = ((it.name||'') + ' ' + (it.notes||'') + ' ' + (it.language||'')).toLowerCase();
        if(!hay.includes(q)) return false;
      }
      return true;
    });
    renderList(filtered);
  }

  // init
  document.addEventListener('DOMContentLoaded', async () => {
    await loadLanguages();
    await fetchList();
  });

  // UI wiring
  refreshBtn.addEventListener('click', async ()=> { await loadLanguages(); await fetchList(); });
  addBtn.addEventListener('click', openAdd);

})();
