/***************************************************
 * BiggBoss Dashboard — Seasons Panel (Server Mode)
 * CLEAN JS FILE — NO UI CHANGES — 2025 VERSION
 **************************************************/

// API endpoints
const API_SEASONS = "api/get_seasons.php";
const API_SAVE    = "api/save_season.php";
const API_UPDATE  = "api/update_season.php";
const API_DELETE  = "api/delete_season.php";
const API_LANG    = "api/get_languages.php";
const API_HOST    = "api/get_hosts.php";

// DOM elements
const filterLanguage = document.getElementById('filterLanguage');
const filterShowType = document.getElementById('filterShowType');
const searchSeason   = document.getElementById('searchSeason');
const searchBtn      = document.getElementById('searchBtn');
const createSeasonBtn= document.getElementById('createSeasonBtn');
const languagesArea  = document.getElementById('languagesArea');
const noSeasons      = document.getElementById('noSeasons');

const seasonModal    = document.getElementById('seasonModal');
const closeSeasonModal = document.getElementById('closeSeasonModal');
const modalTitle     = document.getElementById('modalTitle');

// modal form fields
const m_show_type       = document.getElementById('m_show_type');
const m_show_custom     = document.getElementById('m_show_custom');
const m_show_custom_wrap= document.getElementById('m_show_custom_wrap');
const m_language        = document.getElementById('m_language');
const m_season_no       = document.getElementById('m_season_no');
const m_tag             = document.getElementById('m_tag');
const m_year            = document.getElementById('m_year');
const m_notes           = document.getElementById('m_notes');
const m_logo_input      = document.getElementById('m_logo_input');
const m_logo_preview    = document.getElementById('m_logo_preview');
const m_logo_placeholder= document.getElementById('m_logo_placeholder');
const m_trophy_input    = document.getElementById('m_trophy_input');
const m_trophy_preview  = document.getElementById('m_trophy_preview');
const m_trophy_placeholder = document.getElementById('m_trophy_placeholder');
const m_house_input     = document.getElementById('m_house_input');
const m_house_gallery   = document.getElementById('m_house_gallery');
const m_host            = document.getElementById('m_host');
const participantsList  = document.getElementById('participantsList');
const saveSeasonBtn     = document.getElementById('saveSeasonBtn');

// view modal
const viewModal         = document.getElementById('viewModal');
const view_media_img    = document.getElementById('view_media_img');
const view_title        = document.getElementById('view_title');
const view_meta_row     = document.getElementById('view_meta_row');
const view_description  = document.getElementById('view_description');
const view_placements   = document.getElementById('view_placements');
const view_ott_logo     = document.getElementById('view_ott_logo');
const view_ott_name     = document.getElementById('view_ott_name');
const view_ott_link     = document.getElementById('view_ott_link');
const view_sat_logo     = document.getElementById('view_sat_logo');
const view_sat_name     = document.getElementById('view_sat_name');
const view_notes        = document.getElementById('view_notes');

// placement selectors
const p_winner = document.getElementById('p_winner');
const p_ru1    = document.getElementById('p_ru1');
const p_ru2    = document.getElementById('p_ru2');
const p_ru3    = document.getElementById('p_ru3');
const p_ru4    = document.getElementById('p_ru4');
const autoFillBtn = document.getElementById('autoFillPlacements');

// global state
let serverSeasons = [];
let hostsCache = [];
let languagesCache = [];
let contestantsCache = []; 
let editing = null;
let logoData = "";
let trophyData = "";
let housePhotos = [];
let selectedParticipants = new Set();

/************* Helper functions *************/
function escapeHtml(s){
    if(!s) return "";
    return String(s).replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;");
}

async function fetchJSON(url, options={}){
    const res = await fetch(url, options);
    return res.json();
}
/***************************************************
 * Load languages + hosts from server
 **************************************************/
async function loadLanguagesAndHosts(){
    try {
        const langRes = await fetchJSON(API_LANG);
        const hostRes = await fetchJSON(API_HOST);

        languagesCache = langRes.data || [];
        hostsCache = hostRes.data || [];

        // fill language dropdowns
        filterLanguage.innerHTML = '<option value="">All languages</option>';
        m_language.innerHTML = '<option value="">-- Select language --</option>';

        languagesCache.forEach(l => {
            filterLanguage.innerHTML += `<option value="${escapeHtml(l.name)}">${escapeHtml(l.name)}</option>`;
            m_language.innerHTML += `<option value="${escapeHtml(l.name)}">${escapeHtml(l.name)}</option>`;
        });

        // fill hosts
        m_host.innerHTML = '<option value="">-- Select host --</option>';
        hostsCache.forEach(h => {
            m_host.innerHTML += `<option value="${h.id}">${escapeHtml(h.name)}</option>`;
        });

        populateShowTypeFilter();

    } catch (e) {
        console.error("Failed to load languages/hosts", e);
    }
}

/***************************************************
 * Load languages + hosts from server
 **************************************************/
async function loadLanguagesAndHosts(){
    try {
        const langRes = await fetchJSON(API_LANG);
        const hostRes = await fetchJSON(API_HOST);

        languagesCache = langRes.data || [];
        hostsCache = hostRes.data || [];

        // fill language dropdowns
        filterLanguage.innerHTML = '<option value="">All languages</option>';
        m_language.innerHTML = '<option value="">-- Select language --</option>';

        languagesCache.forEach(l => {
            filterLanguage.innerHTML += `<option value="${escapeHtml(l.name)}">${escapeHtml(l.name)}</option>`;
            m_language.innerHTML += `<option value="${escapeHtml(l.name)}">${escapeHtml(l.name)}</option>`;
        });

        // fill hosts
        m_host.innerHTML = '<option value="">-- Select host --</option>';
        hostsCache.forEach(h => {
            m_host.innerHTML += `<option value="${h.id}">${escapeHtml(h.name)}</option>`;
        });

        populateShowTypeFilter();

    } catch (e) {
        console.error("Failed to load languages/hosts", e);
    }
}

/***************************************************
 * Load seasons from server (instead of localStorage)
 **************************************************/
async function loadSeasonsFromServer(){
    try {
        const res = await fetchJSON(API_SEASONS);
        serverSeasons = res.data || [];
        renderLanguageTiles();
    } catch (e) {
        console.error("Failed to load seasons", e);
    }
}

/***************************************************
 * Show Type Filter (Dynamic based on server data)
 **************************************************/
function populateShowTypeFilter(){
    const defaultTypes = [
        {val:'', label:'All types'},
        {val:'season', label:'BiggBoss Season'},
        {val:'ott', label:'BiggBoss OTT'},
        {val:'ultimate', label:'BiggBoss Ultimate'}
    ];

    const seen = new Map();
    serverSeasons.forEach(s => {
        const key = (s.show_type || "").toLowerCase();
        const label = s.show_type_label || s.show_type || key;
        if(key) seen.set(key, label);
    });

    filterShowType.innerHTML = "";
    defaultTypes.forEach(d => {
        filterShowType.innerHTML += `<option value="${d.val}">${d.label}</option>`;
    });

    seen.forEach((label, key) => {
        if(["season","ott","ultimate"].includes(key)) return;
        filterShowType.innerHTML += `<option value="${key}">${label}</option>`;
    });
}

/***************************************************
 * View Modal
 **************************************************/
function openViewModal(id){
    const s = serverSeasons.find(x => x.id == id);
    if(!s) return;

    const thumb = s.logo || s.trophy || (s.house_photos?.[0] || "");
    if(thumb){
        view_media_img.src = thumb;
        view_media_img.style.display = "block";
    } else {
        view_media_img.style.display = "none";
    }

    const label = s.show_type_label || s.show_type || "season";

    view_title.textContent = `${s.tag} (${label})`;

    view_meta_row.innerHTML = `
        <div class="chip">Language: <strong>${escapeHtml(s.language)}</strong></div>
        <div class="chip">Season No: <strong>${escapeHtml(s.season_number)}</strong></div>
        <div class="chip">Year: <strong>${escapeHtml(s.year)}</strong></div>
        <div class="chip">Type: <strong>${escapeHtml(label)}</strong></div>
        <div class="chip">Participants: <strong>${(s.participants||[]).length}</strong></div>
    `;

    view_description.textContent = s.description || "";

    // placements
    view_placements.innerHTML = "";
    const places = [
        {k:"winner", label:"Winner"},
        {k:"ru1", label:"1st Runner-up"},
        {k:"ru2", label:"2nd Runner-up"},
        {k:"ru3", label:"3rd Runner-up"},
        {k:"ru4", label:"4th Runner-up"}
    ];

    places.forEach(p => {
        const val = s.placements?.[p.k] || "";
        const box = document.createElement("div");
        box.className = "placement-box";
        box.innerHTML = `<h5>${p.label}</h5><p>${escapeHtml(val)}</p>`;
        view_placements.appendChild(box);
    });

    // broadcasts
    if(s.broadcasts?.ott){
        const o = s.broadcasts.ott;
        view_ott_name.textContent = o.name || "-";
        if(o.link){
            view_ott_link.href = o.link;
            view_ott_link.textContent = o.link;
            view_ott_link.style.display = "block";
        } else view_ott_link.style.display = "none";
    }

    if(s.broadcasts?.satellite){
        view_sat_name.textContent = s.broadcasts.satellite.app_name || "-";
    }

    view_notes.innerHTML = `<strong>Notes</strong><div>${escapeHtml(s.description || "-")}</div>`;

    viewModal.classList.add("open");
    document.body.style.overflow = "hidden";
}

document.getElementById("closeView").onclick = () => {
    viewModal.classList.remove("open");
    document.body.style.overflow = "";
};
/***************************************************
 * House Photos Gallery Renderer
 **************************************************/
function renderHouseGallery(){
    m_house_gallery.innerHTML = "";
    housePhotos.forEach((src, idx) => {
        const box = document.createElement("div");
        box.style.position = "relative";
        box.style.display = "inline-block";
        box.style.marginRight = "8px";

        const img = document.createElement("img");
        img.src = src;
        img.style.width = "72px";
        img.style.height = "72px";
        img.style.objectFit = "cover";
        img.style.borderRadius = "8px";

        const del = document.createElement("div");
        del.textContent = "✕";
        del.style.position = "absolute";
        del.style.top = "-6px";
        del.style.right = "-6px";
        del.style.background = "#fff";
        del.style.border = "1px solid #ccc";
        del.style.borderRadius = "50%";
        del.style.padding = "2px 6px";
        del.style.cursor = "pointer";
        del.onclick = () => {
            housePhotos.splice(idx, 1);
            renderHouseGallery();
        };

        box.appendChild(img);
        box.appendChild(del);
        m_house_gallery.appendChild(box);
    });
}

/***************************************************
 * Upload image → returns URL (server stored image)
 **************************************************/
async function uploadImage(file){
    const fd = new FormData();
    fd.append("file", file);

    const res = await fetch("api/upload_image.php", {
        method: "POST",
        body: fd
    });

    const json = await res.json();
    if(json.status === "success") return json.url;
    return "";
}

/***************************************************
 * Open Edit Modal
 **************************************************/
function openEditSeason(id){
    const s = serverSeasons.find(x => x.id == id);
    if(!s) return;

    editing = id;

    modalTitle.textContent = "Edit Season";

    // show type
    const st = (s.show_type || "").toLowerCase();
    if(["season","ott","ultimate"].includes(st)){
        m_show_type.value = st;
        m_show_custom_wrap.style.display = "none";
        m_show_custom.value = "";
    } else {
        m_show_type.value = "custom";
        m_show_custom_wrap.style.display = "block";
        m_show_custom.value = s.show_type_label || s.show_type;
    }

    m_language.value = s.language || "";
    m_season_no.value = s.season_number || "";
    m_tag.value = s.tag || "";
    m_year.value = s.year || "";
    m_notes.value = s.description || "";

    // logo
    logoData = s.logo || "";
    if(logoData){
        m_logo_preview.src = logoData;
        m_logo_preview.style.display = "block";
        m_logo_placeholder.style.display = "none";
    } else {
        m_logo_preview.style.display = "none";
        m_logo_placeholder.style.display = "block";
    }

    // trophy
    trophyData = s.trophy || "";
    if(trophyData){
        m_trophy_preview.src = trophyData;
        m_trophy_preview.style.display = "block";
        m_trophy_placeholder.style.display = "none";
    } else {
        m_trophy_preview.style.display = "none";
        m_trophy_placeholder.style.display = "block";
    }

    // house photos
    housePhotos = s.house_photos || [];
    renderHouseGallery();

    // participants
    selectedParticipants = new Set(s.participants || []);
    renderParticipants(Array.from(selectedParticipants));

    // broadcasts
    document.getElementById("b_ott_name").value = s.broadcasts?.ott?.name || "";
    document.getElementById("b_ott_link").value = s.broadcasts?.ott?.link || "";
    document.getElementById("b_sat_name").value = s.broadcasts?.satellite?.app_name || "";

    // placements
    const pl = s.placements || {};
    populatePlacementSelects([
        pl.winner, pl.ru1, pl.ru2, pl.ru3, pl.ru4
    ]);

    // host
    if(s.host_id) m_host.value = s.host_id;

    openModal();
}

/***************************************************
 * Open Create Modal
 **************************************************/
createSeasonBtn.onclick = () => {
    editing = null;
    modalTitle.textContent = "Create Season";

    m_show_type.value = "season";
    m_show_custom.value = "";
    m_show_custom_wrap.style.display = "none";

    m_language.value = "";
    m_season_no.value = "";
    m_tag.value = "";
    m_year.value = "";
    m_notes.value = "";

    logoData = "";
    trophyData = "";
    housePhotos = [];

    m_logo_preview.style.display = "none";
    m_logo_placeholder.style.display = "block";
    m_trophy_preview.style.display = "none";
    m_trophy_placeholder.style.display = "block";

    renderHouseGallery();
    selectedParticipants = new Set();
    renderParticipants([]);

    document.getElementById("b_ott_name").value = "";
    document.getElementById("b_ott_link").value = "";
    document.getElementById("b_sat_name").value = "";

    loadLanguagesAndHosts();

    openModal();
};
/***************************************************
 * Modal Open / Close
 **************************************************/
function openModal(){
    seasonModal.classList.add("open");
    document.body.style.overflow = "hidden";
}

function closeModal(){
    seasonModal.classList.remove("open");
    document.body.style.overflow = "";
}

closeSeasonModal.onclick = closeModal;
seasonModal.onclick = (e) => {
    if(e.target === seasonModal) closeModal();
};

/***************************************************
 * PARTICIPANTS: Filter based on Show Type & Season No
 **************************************************/
function getFilteredContestantsForModal() {
    if(!contestantsCache.length) return [];

    const showType = (m_show_type.value === "custom")
        ? m_show_custom.value.toLowerCase().trim()
        : m_show_type.value.toLowerCase().trim();

    const seasonNo = (m_season_no.value || "").trim();

    return contestantsCache.filter(c => {
        const c_show = (c.show_type || "").toLowerCase();
        const c_season = String(c.season_no || "").trim();

        // Match season
        if(seasonNo && c_season !== seasonNo) return false;

        // match show type
        if(showType && c_show !== showType) return false;

        return true;
    });
}

function renderParticipants(selectedIds){
    selectedParticipants = new Set(selectedIds || []);
    participantsList.innerHTML = "";

    const filtered = getFilteredContestantsForModal();
    if(!filtered.length){
        participantsList.innerHTML =
          `<div class="text-muted" style="padding:8px;">
             No contestants match this Show Type / Season No.
           </div>`;
        populatePlacementSelects([], []);
        return;
    }

    filtered.sort((a,b)=> (a.name||"").localeCompare(b.name||""));

    filtered.forEach(c => {
        const row = document.createElement("div");
        row.className = "participant-item";

        const left = document.createElement("div");
        left.innerHTML = `
            <div style="font-weight:600;">${escapeHtml(c.name)}</div>
            <div class="text-muted" style="font-size:12px;">${escapeHtml(c.language||"")}</div>
        `;

        const checkWrap = document.createElement("div");
        const cb = document.createElement("input");
        cb.type = "checkbox";
        cb.value = c.id;
        cb.checked = selectedParticipants.has(c.id);

        cb.onchange = () => {
            if(cb.checked) selectedParticipants.add(c.id);
            else selectedParticipants.delete(c.id);
        };

        checkWrap.appendChild(cb);
        row.appendChild(left);
        row.appendChild(checkWrap);

        participantsList.appendChild(row);
    });

    populatePlacementSelects(Array.from(selectedParticipants), filtered);
}

/***************************************************
 * Placement Select Builder
 **************************************************/
function populatePlacementSelects(selectedIds, contestantsList){
    const list = contestantsList || [];
    const opts = [{id:"", name:"-- None --"}].concat(
        list.map(c => ({id:c.id, name:c.name}))
    );

    const selects = [p_winner, p_ru1, p_ru2, p_ru3, p_ru4];

    selects.forEach(sel => {
        sel.innerHTML = "";
        opts.forEach(o => {
            sel.innerHTML += `<option value="${o.id}">${escapeHtml(o.name)}</option>`;
        });
    });

    if(selectedIds){
        p_winner.value = selectedIds[0] || "";
        p_ru1.value = selectedIds[1] || "";
        p_ru2.value = selectedIds[2] || "";
        p_ru3.value = selectedIds[3] || "";
        p_ru4.value = selectedIds[4] || "";
    }
}

/***************************************************
 * Image Inputs → Convert to Base64 or Upload to Server
 **************************************************/
m_logo_input.onchange = async (ev) => {
    const f = ev.target.files[0];
    if(!f) return;

    const url = await uploadImage(f);
    if(url){
        logoData = url;
        m_logo_preview.src = url;
        m_logo_preview.style.display = "block";
        m_logo_placeholder.style.display = "none";
    }
};

m_trophy_input.onchange = async (ev) => {
    const f = ev.target.files[0];
    if(!f) return;

    const url = await uploadImage(f);
    if(url){
        trophyData = url;
        m_trophy_preview.src = url;
        m_trophy_preview.style.display = "block";
        m_trophy_placeholder.style.display = "none";
    }
};

m_house_input.onchange = async (ev) => {
    const files = Array.from(ev.target.files || []);
    for(const f of files){
        const url = await uploadImage(f);
        if(url) housePhotos.push(url);
    }
    renderHouseGallery();
};

/***************************************************
 * Auto-Fill placements
 **************************************************/
autoFillBtn.onclick = () => {
    const arr = Array.from(selectedParticipants || []);
    const filled = arr.slice(0,5);

    p_winner.value = filled[0] || "";
    p_ru1.value = filled[1] || "";
    p_ru2.value = filled[2] || "";
    p_ru3.value = filled[3] || "";
    p_ru4.value = filled[4] || "";
};
/***************************************************
 * SAVE SEASON (Create Mode)
 **************************************************/
async function saveSeasonToServer(data){
    const res = await fetch(API_SAVE, {
        method: "POST",
        headers: {"Content-Type":"application/json"},
        body: JSON.stringify(data)
    });

    return res.json();
}

/***************************************************
 * UPDATE SEASON (Edit Mode)
 **************************************************/
async function updateSeasonToServer(data){
    const res = await fetch(API_UPDATE, {
        method: "POST",
        headers: {"Content-Type":"application/json"},
        body: JSON.stringify(data)
    });

    return res.json();
}

/***************************************************
 * DELETE SEASON
 **************************************************/
async function deleteSeason(id){
    if(!confirm("Delete season?")) return;

    const res = await fetch(API_DELETE, {
        method: "POST",
        headers: {"Content-Type":"application/json"},
        body: JSON.stringify({id})
    });

    const json = await res.json();
    if(json.status === "success"){
        loadSeasonsFromServer();
    } else {
        alert("Error deleting season: " + json.message);
    }
}

/***************************************************
 * SAVE BUTTON HANDLER
 **************************************************/
saveSeasonBtn.onclick = async () => {
    const raw_show_type = m_show_type.value;
    let show_type = raw_show_type;
    let show_type_label = "";

    if(raw_show_type === "custom"){
        if(!m_show_custom.value.trim()){
            alert("Enter custom show name");
            return;
        }
        show_type_label = m_show_custom.value.trim();
        show_type = m_show_custom.value.toLowerCase().replace(/\s+/g,"-");
    } else {
        show_type_label =
            raw_show_type === "season" ? "BiggBoss Season" :
            raw_show_type === "ott"    ? "BiggBoss OTT" :
            raw_show_type === "ultimate" ? "BiggBoss Ultimate" :
            raw_show_type;
    }

    const data = {
        id: editing,
        season_number: m_season_no.value,
        language: m_language.value,
        tag: m_tag.value,
        year: m_year.value,
        description: m_notes.value,
        host_id: m_host.value,
        show_type,
        show_type_label,
        participants: Array.from(selectedParticipants),
        placements: {
            winner: p_winner.value,
            ru1: p_ru1.value,
            ru2: p_ru2.value,
            ru3: p_ru3.value,
            ru4: p_ru4.value
        },
        broadcasts: {
            ott: {
                name: document.getElementById("b_ott_name").value,
                link: document.getElementById("b_ott_link").value
            },
            satellite: {
                app_name: document.getElementById("b_sat_name").value
            }
        },
        logo: logoData,
        trophy_photo: trophyData,
        house_photos: housePhotos
    };

    // Validation
    if(!data.language){
        alert("Select language");
        return;
    }
    if(!data.tag){
        alert("Enter season tag/title");
        return;
    }

    let response;
    if(editing){
        response = await updateSeasonToServer(data);
    } else {
        response = await saveSeasonToServer(data);
    }

    if(response.status === "success"){
        closeModal();
        loadSeasonsFromServer();
    } else {
        alert("Error: " + response.message);
    }
};

/***************************************************
 * Render Language Tiles (Group Seasons)
 **************************************************/
function renderLanguageTiles(){
    languagesArea.innerHTML = "";

    const q = searchSeason.value.toLowerCase();
    const typeFilter = filterShowType.value;
    const langFilter = filterLanguage.value;

    const groups = {};

    serverSeasons.forEach(s => {
        // type filter
        if(typeFilter && String(s.show_type).toLowerCase() !== typeFilter.toLowerCase())
            return;

        // language filter
        if(langFilter && s.language !== langFilter)
            return;

        // search
        const text = (s.tag || "") + " " + (s.season_number || "") + " " + (s.year || "");
        if(q && !text.toLowerCase().includes(q)) return;

        if(!groups[s.language]) groups[s.language] = [];
        groups[s.language].push(s);
    });

    let total = 0;

    Object.keys(groups).sort().forEach(lang => {
        const items = groups[lang];
        total += items.length;

        // main tile
        const tile = document.createElement("div");
        tile.className = "lang-tile";
        tile.innerHTML = `
            <div>
                <h3>${escapeHtml(lang)}</h3>
                <div class="meta">${items.length} season(s)</div>
            </div>
            <div class="chev">▸</div>
        `;

        const content = document.createElement("div");
        content.className = "lang-content";
        content.innerHTML = `<div class="season-list"></div>`;
        const list = content.querySelector(".season-list");

        items.sort((a,b) => Number(a.season_number) - Number(b.season_number));

        items.forEach(s => {
            const card = document.createElement("div");
            card.className = "season-card";

            const thumb = s.logo || s.trophy || (s.house_photos?.[0] || "");

            let badge = "";
            if(s.show_type === "season") badge = '<span class="show-type-badge show-type-season">Season</span>';
            else if(s.show_type === "ott") badge = '<span class="show-type-badge show-type-ott">OTT</span>';
            else if(s.show_type === "ultimate") badge = '<span class="show-type-badge show-type-ultimate">Ultimate</span>';
            else badge = `<span class="show-type-badge show-type-custom">${escapeHtml(s.show_type_label)}</span>`;

            const hostName = hostsCache.find(h => h.id == s.host_id)?.name || "-";

            card.innerHTML = `
                <div class="season-media">
                    <img src="${thumb}" onerror="this.style.display='none'">
                </div>

                <div class="season-main">
                    <div style="display:flex;justify-content:space-between;">
                        <div class="season-title">
                            ${escapeHtml(s.tag)} ${badge}
                        </div>
                    </div>

                    <div class="season-meta-line">
                        <div class="chip">Language: <strong>${escapeHtml(s.language)}</strong></div>
                        <div class="chip">Season No: <strong>${escapeHtml(s.season_number)}</strong></div>
                        <div class="chip">Host: <strong>${escapeHtml(hostName)}</strong></div>
                        <div class="chip">Participants: <strong>${(s.participants||[]).length}</strong></div>
                    </div>

                    <div style="margin-top:10px; display:flex; justify-content:flex-end;">
                        <div class="action-row">
                            <button class="btn btn-ghost" onclick="openViewModal('${s.id}')">View</button>
                            <button class="btn btn-ghost" onclick="openEditSeason('${s.id}')">Edit</button>
                            <button class="btn btn-danger" onclick="deleteSeason('${s.id}')">Delete</button>
                        </div>
                    </div>
                </div>
            `;

            list.appendChild(card);
        });

        tile.onclick = (e) => {
            if(e.target.closest(".btn")) return;

            const open = tile.classList.contains("open");
            document.querySelectorAll(".lang-tile.open").forEach(t => {
                t.classList.remove("open");
                t.nextElementSibling.style.display = "none";
                t.querySelector(".chev").textContent = "▸";
            });

            if(!open){
                tile.classList.add("open");
                content.style.display = "block";
                tile.querySelector(".chev").textContent = "▾";
            }
        };

        languagesArea.appendChild(tile);
        languagesArea.appendChild(content);
    });

    noSeasons.style.display = total ? "none" : "block";
}
/***************************************************
 * SEARCH, FILTER EVENTS
 **************************************************/
searchBtn.onclick = () => renderLanguageTiles();
filterLanguage.onchange = () => renderLanguageTiles();
filterShowType.onchange = () => renderLanguageTiles();
searchSeason.onkeyup = (e) => {
    if(e.key === "Enter") renderLanguageTiles();
};

/***************************************************
 * LIVE UPDATE of Participants when:
 * - Show Type changes
 * - Custom show type input changes
 * - Season number changes
 **************************************************/
m_show_type.onchange = () => {
    if(m_show_type.value === "custom"){
        m_show_custom_wrap.style.display = "block";
    } else {
        m_show_custom_wrap.style.display = "none";
        m_show_custom.value = "";
    }
    renderParticipants(Array.from(selectedParticipants));
};

m_show_custom.oninput = () => {
    renderParticipants(Array.from(selectedParticipants));
};

m_season_no.oninput = () => {
    renderParticipants(Array.from(selectedParticipants));
};

m_tag.oninput = () => {
    // not required for filtering participants,
    // but kept for compatibility with your old UI flow
};

/***************************************************
 * INITIAL DATA LOAD
 **************************************************/
async function initSeasonsPage(){
    await loadLanguagesAndHosts();
    await loadSeasonsFromServer();
    await loadContestants();   // load contestants from your API when available
    renderParticipants([]);
}

/***************************************************
 * Placeholder: Load contestants from API
 * You will add real API later:
 * /dashboard/api/get_contestants.php
 **************************************************/
async function loadContestants(){
    // until you create API, use empty array
    contestantsCache = []; 
}

/***************************************************
 * INIT
 **************************************************/
setTimeout(() => {
    initSeasonsPage();
}, 200);

