/***************************************************
 * profiles.js — Path: /public_html/dashboard/profiles.js
 ***************************************************/

const API_GET_PROFILES = "api/get_profiles.php";
const API_SAVE_PROFILE = "api/save_profile.php";
const API_UPDATE_PROFILE = "api/update_profile.php";
const API_DELETE_PROFILE = "api/delete_profile.php";
const API_UPLOAD = "api/upload_image.php";
const API_LANG = "api/get_languages.php";

const profileArea = document.getElementById('profileArea');
const addProfileBtn = document.getElementById('addProfileBtn');
const noProfiles = document.getElementById('noProfiles');

const profileModal = document.getElementById('profileModal');
const closeProfileModal = document.getElementById('closeProfileModal');
const profileModalTitle = document.getElementById('profileModalTitle');

const p_name = document.getElementById('p_name');
const p_language = document.getElementById('p_language');
const p_age = document.getElementById('p_age');
const p_description = document.getElementById('p_description');
const p_photo_input = document.getElementById('p_photo_input');
const p_photo_preview = document.getElementById('p_photo_preview');
const p_photo_placeholder = document.getElementById('p_photo_placeholder');
const p_gallery_input = document.getElementById('p_gallery_input');
const p_gallery_view = document.getElementById('p_gallery_view');
const p_social_instagram = document.getElementById('p_social_instagram');
const p_social_twitter = document.getElementById('p_social_twitter');
const p_social_youtube = document.getElementById('p_social_youtube');
const saveProfileBtn = document.getElementById('saveProfileBtn');

let languages = [];
let profiles = [];
let editingId = null;
let photoUrl = "";
let galleryUrls = [];

/* Helpers */
async function fetchJSON(url, opts={}){ const r = await fetch(url, opts); return r.json(); }
function escapeHtml(x){ if(!x) return ""; return String(x).replace(/</g,"&lt;").replace(/>/g,"&gt;"); }

/* Load languages */
async function loadLanguages(){
  try{
    const r = await fetchJSON(API_LANG);
    languages = r.data || [];
  }catch(e){ languages = []; }

  p_language.innerHTML = `<option value="">-- Select --</option>`;
  languages.forEach(l=>{
    p_language.innerHTML += `<option value="${escapeHtml(l.name)}">${escapeHtml(l.name)}</option>`;
  });
}

/* Load profiles */
async function loadProfiles(){
  try{
    const r = await fetchJSON(API_GET_PROFILES);
    profiles = r.data || [];
  }catch(e){ profiles=[]; }

  renderProfiles();
}

/* Render */
function renderProfiles(){
  profileArea.innerHTML = "";

  if(!profiles.length){
    noProfiles.style.display = "block";
    return;
  }
  noProfiles.style.display = "none";

  profiles.forEach(p=>{
    const card = document.createElement('div');
    card.className = "card";

    const imgWrap = document.createElement('div');
    imgWrap.className = "card-img";

    const img = document.createElement('img');
    img.src = p.photo || "";
    img.onerror = ()=> img.style.display='none';
    imgWrap.appendChild(img);

    const body = document.createElement('div');
    body.className = "card-body";

    body.innerHTML = `
      <div style="display:flex;justify-content:space-between;">
        <div style="font-weight:700;">${escapeHtml(p.name)}</div>
        <div style="display:flex;gap:8px;">
          <button class="btn btn-ghost" onclick="openEdit(${p.id})">Edit</button>
          <button class="btn btn-danger" onclick="delProfile(${p.id})">Delete</button>
        </div>
      </div>
      <div class="meta">Language: ${escapeHtml(p.language)}</div>
      <div class="meta">Age: ${escapeHtml(p.age)}</div>
      <div style="margin-top:6px; font-size:13px;">${escapeHtml(p.description)}</div>
    `;

    card.appendChild(imgWrap);
    card.appendChild(body);

    profileArea.appendChild(card);
  });
}

/* Open/close modal */
function openModal(){ profileModal.classList.add("open"); document.body.style.overflow="hidden"; }
function closeModal(){ profileModal.classList.remove("open"); document.body.style.overflow=""; }
closeProfileModal.onclick = closeModal;

profileModal.onclick = (e)=>{ if(e.target === profileModal) closeModal(); };

/* Add profile */
addProfileBtn.onclick = ()=>{
  editingId = null;
  profileModalTitle.textContent = "Add Profile";
  p_name.value = ""; p_language.value=""; p_age.value=""; p_description.value="";
  p_social_instagram.value = ""; p_social_twitter.value=""; p_social_youtube.value="";
  photoUrl = ""; p_photo_preview.style.display="none"; p_photo_placeholder.style.display="block";
  galleryUrls = []; p_gallery_view.innerHTML="";
  openModal();
};

/* Photo upload */
p_photo_input.onchange = async (e)=>{
  const f = e.target.files[0];
  if(!f) return;
  const fd = new FormData();
  fd.append("file", f);

  const r = await fetchJSON(API_UPLOAD, { method:"POST", body:fd });
  if(r.status==="success"){
    photoUrl = r.url || ("uploads/" + r.url);
    p_photo_preview.src = photoUrl;
    p_photo_preview.style.display="block";
    p_photo_placeholder.style.display="none";
  }
};

/* Gallery upload */
p_gallery_input.onchange = async (e)=>{
  const files = [...e.target.files];
  for(const f of files){
    const fd = new FormData(); fd.append("file", f);
    const r = await fetchJSON(API_UPLOAD, { method:"POST", body:fd });
    if(r.status==="success"){
      const url = r.url || ("uploads/" + r.url);
      galleryUrls.push(url);
      const img = document.createElement("img");
      img.src = url;
      p_gallery_view.appendChild(img);
    }
  }
};

/* Save profile */
saveProfileBtn.onclick = async ()=>{
  const payload = {
    name: p_name.value.trim(),
    language: p_language.value.trim(),
    age: p_age.value.trim(),
    photo: photoUrl,
    description: p_description.value.trim(),
    social: {
      instagram: p_social_instagram.value.trim(),
      twitter: p_social_twitter.value.trim(),
      youtube: p_social_youtube.value.trim()
    },
    gallery: galleryUrls
  };

  let r;
  if(editingId){
    payload.id = editingId;
    r = await fetchJSON(API_UPDATE_PROFILE, {
      method:"POST",
      headers:{"Content-Type":"application/json"},
      body:JSON.stringify(payload)
    });
  } else {
    r = await fetchJSON(API_SAVE_PROFILE, {
      method:"POST",
      headers:{"Content-Type":"application/json"},
      body:JSON.stringify(payload)
    });
  }

  if(r.status==="success"){
    closeModal();
    loadProfiles();
  } else alert("Error: "+r.message);
};

/* Edit profile */
window.openEdit = async function(id){
  const p = profiles.find(x=>String(x.id)===String(id));
  if(!p) return;

  editingId = id;
  profileModalTitle.textContent = "Edit Profile";

  p_name.value = p.name;
  p_language.value = p.language;
  p_age.value = p.age;
  p_description.value = p.description;

  p_social_instagram.value = p.social?.instagram || "";
  p_social_twitter.value = p.social?.twitter || "";
  p_social_youtube.value = p.social?.youtube || "";

  photoUrl = p.photo || "";
  if(photoUrl){
    p_photo_preview.src = photoUrl;
    p_photo_preview.style.display="block";
    p_photo_placeholder.style.display="none";
  }

  galleryUrls = p.gallery || [];
  p_gallery_view.innerHTML = "";
  galleryUrls.forEach(url=>{
    const img = document.createElement("img");
    img.src = url;
    p_gallery_view.appendChild(img);
  });

  openModal();
};

/* Delete profile */
window.delProfile = async function(id){
  if(!confirm("Delete profile?")) return;
  const r = await fetchJSON(API_DELETE_PROFILE, {
    method:"POST",
    headers:{"Content-Type":"application/json"},
    body:JSON.stringify({id})
  });
  if(r.status==="success") loadProfiles();
};

/* init */
(async ()=>{
  await loadLanguages();
  await loadProfiles();
})();
