/*******************************************************
 * hosts.js  — Host page JS
 * Path: /public_html/dashboard/hosts.js
 *******************************************************/

const API_GET_HOSTS = "api/get_hosts.php";
const API_SAVE_HOST = "api/save_host.php";
const API_UPDATE_HOST = "api/update_host.php";
const API_DELETE_HOST = "api/delete_host.php";
const API_UPLOAD = "api/upload_image.php";
const API_GET_LANG = "api/get_languages.php";

const hostsArea = document.getElementById('hostsArea');
const noHosts = document.getElementById('noHosts');
const addHostBtn = document.getElementById('addHostBtn');

const hostModal = document.getElementById('hostModal');
const closeHostModal = document.getElementById('closeHostModal');
const hostModalTitle = document.getElementById('hostModalTitle');

const h_name = document.getElementById('h_name');
const h_language = document.getElementById('h_language');
const h_working_years = document.getElementById('h_working_years');
const h_bio = document.getElementById('h_bio');
const h_photo_input = document.getElementById('h_photo_input');
const h_photo_preview = document.getElementById('h_photo_preview');
const h_photo_placeholder = document.getElementById('h_photo_placeholder');

const saveHostBtn = document.getElementById('saveHostBtn');

let hosts = [];
let languages = [];
let editingId = null;
let photoUrl = "";

/* Helper */
async function fetchJSON(url, opts){ const r = await fetch(url, opts); return r.json(); }
function escapeHtml(s){ if(!s) return ''; return String(s).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;'); }
function openModal(){ hostModal.classList.add('open'); hostModal.setAttribute('aria-hidden','false'); document.body.style.overflow='hidden'; }
function closeModal(){ hostModal.classList.remove('open'); hostModal.setAttribute('aria-hidden','true'); document.body.style.overflow=''; }

/* Load languages */
async function loadLanguages(){
  try{
    const res = await fetchJSON(API_GET_LANG);
    languages = res.data || [];
  }catch(e){ languages = []; }
  h_language.innerHTML = `<option value="">-- Select language --</option>`;
  languages.forEach(l=>{
    const opt = document.createElement('option'); opt.value = l.name; opt.textContent = l.name; h_language.appendChild(opt);
  });
}

/* Load hosts */
async function loadHosts(){
  try {
    const r = await fetchJSON(API_GET_HOSTS);
    hosts = r.data || [];
  } catch(e){
    hosts = [];
  }
  renderHosts();
}

function renderHosts(){
  hostsArea.innerHTML = "";
  if(!hosts.length){ noHosts.style.display = 'block'; return; }
  noHosts.style.display = 'none';

  hosts.forEach(h=>{
    const card = document.createElement('div');
    card.className = 'host-card';

    const thumb = document.createElement('div'); thumb.className = 'host-thumb';
    const img = document.createElement('img'); img.src = h.photo || ''; img.onerror = ()=> img.style.display='none'; thumb.appendChild(img);

    const info = document.createElement('div');
    info.style.flex = '1';
    info.innerHTML = `<div style="font-weight:700">${escapeHtml(h.name)} ${h.working_years?('(' + escapeHtml(h.working_years) + ')') : ''}</div>
                      <div class="text-muted">${escapeHtml(h.language || '')}</div>
                      <div style="margin-top:8px; font-size:13px;">${escapeHtml(h.bio || '')}</div>`;

    const actions = document.createElement('div'); actions.className = 'actions';
    const btnView = document.createElement('button'); btnView.className='btn btn-ghost'; btnView.textContent='View';
    btnView.addEventListener('click', ()=> openView(h.id));
    const btnEdit = document.createElement('button'); btnEdit.className='btn btn-ghost'; btnEdit.textContent='Edit';
    btnEdit.addEventListener('click', ()=> openEdit(h.id));
    const btnDelete = document.createElement('button'); btnDelete.className='btn btn-danger'; btnDelete.textContent='Delete';
    btnDelete.addEventListener('click', ()=> deleteHost(h.id));
    actions.appendChild(btnView); actions.appendChild(btnEdit); actions.appendChild(btnDelete);

    card.appendChild(thumb);
    card.appendChild(info);
    card.appendChild(actions);

    hostsArea.appendChild(card);
  });
}

/* Add new host */
addHostBtn.addEventListener('click', ()=>{
  editingId = null; photoUrl = "";
  hostModalTitle.textContent = 'Add Host';
  h_name.value = ""; h_language.value = ""; h_working_years.value = ""; h_bio.value = "";
  h_photo_preview.style.display = 'none'; h_photo_placeholder.style.display = 'block';
  openModal();
});

/* close */
closeHostModal.addEventListener('click', closeModal);
hostModal.addEventListener('click', (e)=>{ if(e.target === hostModal) closeModal(); });

/* Photo upload */
h_photo_input.addEventListener('change', async (e)=>{
  const f = e.target.files[0]; if(!f) return;
  const fd = new FormData(); fd.append('file', f);
  const res = await fetchJSON(API_UPLOAD, { method:'POST', body: fd });
  if(res.status === 'success'){
    photoUrl = res.url || res.path || res.filename || res.file || res.data || res;
    // try a few variations if upload returns different keys
    if(typeof photoUrl === 'object') photoUrl = photoUrl.url || photoUrl.path || '';
    if(!photoUrl) photoUrl = (res.url ? res.url : '');
    h_photo_preview.src = photoUrl;
    h_photo_preview.style.display = 'block';
    h_photo_placeholder.style.display = 'none';
  } else {
    alert(res.message || 'Upload failed');
  }
});

/* Save host */
saveHostBtn.addEventListener('click', async ()=>{
  const payload = {
    name: h_name.value.trim(),
    language: h_language.value,
    working_years: h_working_years.value.trim(),
    bio: h_bio.value.trim(),
    photo: photoUrl || ''
  };

  let res;
  if(editingId){
    payload.id = editingId;
    res = await fetchJSON(API_UPDATE_HOST, {
      method: 'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify(payload)
    });
  } else {
    res = await fetchJSON(API_SAVE_HOST, {
      method: 'POST',
      headers: {'Content-Type':'application/json'},
      body: JSON.stringify(payload)
    });
  }

  if(res.status === 'success'){
    closeModal();
    await loadHosts();
  } else {
    alert(res.message || 'Error');
  }
});

/* open edit */
function openEdit(id){
  const h = hosts.find(x=>String(x.id) === String(id));
  if(!h) return alert('Host not found');
  editingId = id;
  hostModalTitle.textContent = 'Edit Host';
  h_name.value = h.name || '';
  h_language.value = h.language || '';
  h_working_years.value = h.working_years || '';
  h_bio.value = h.bio || '';
  photoUrl = h.photo || '';
  if(photoUrl){ h_photo_preview.src = photoUrl; h_photo_preview.style.display='block'; h_photo_placeholder.style.display='none'; } else { h_photo_preview.style.display='none'; h_photo_placeholder.style.display='block'; }
  openModal();
}

/* view (simple modal or alert) */
function openView(id){
  const h = hosts.find(x=>String(x.id) === String(id));
  if(!h) return;
  // simple view — reuse modal
  editingId = null;
  hostModalTitle.textContent = 'Host: ' + (h.name || '');
  h_name.value = h.name || '';
  h_language.value = h.language || '';
  h_working_years.value = h.working_years || '';
  h_bio.value = h.bio || '';
  photoUrl = h.photo || '';
  if(photoUrl){ h_photo_preview.src = photoUrl; h_photo_preview.style.display='block'; h_photo_placeholder.style.display='none'; } else { h_photo_preview.style.display='none'; h_photo_placeholder.style.display='block'; }
  // disable inputs for view-only
  Array.from([h_name, h_language, h_working_years, h_bio, h_photo_input]).forEach(e=> e.disabled = true);
  saveHostBtn.style.display = 'none';
  openModal();

  // when closing, re-enable
  const restore = ()=> {
    Array.from([h_name, h_language, h_working_years, h_bio, h_photo_input]).forEach(e=> e.disabled = false);
    saveHostBtn.style.display = '';
    closeHostModal.removeEventListener('click', restore);
  };
  closeHostModal.addEventListener('click', restore);
}

/* delete host */
async function deleteHost(id){
  if(!confirm('Delete host?')) return;
  const res = await fetchJSON(API_DELETE_HOST, {
    method:'POST',
    headers:{'Content-Type':'application/json'},
    body: JSON.stringify({id})
  });
  if(res.status === 'success') loadHosts(); else alert(res.message || 'Delete failed');
}

/* init */
(async ()=>{
  await loadLanguages();
  await loadHosts();
})();
