/***************************************************
 * contestants.js — BiggBoss Admin Panel
 * Path: /public_html/dashboard/contestants.js
 ***************************************************/

// API endpoints
const API_GET_CONTESTANTS = "api/get_contestants.php";
const API_SAVE_CONTESTANT = "api/save_contestant.php";
const API_UPDATE_CONTESTANT = "api/update_contestant.php";
const API_DELETE_CONTESTANT = "api/delete_contestant.php";
const API_UPLOAD_IMAGE = "api/upload_image.php";
const API_LANG = "api/get_languages.php";


// DOM refs
const filterLanguage = document.getElementById('filterLanguage');
const searchBox = document.getElementById('searchBox');
const searchBtn = document.getElementById('searchBtn');
const createBtn = document.getElementById('createBtn');
const contestantsArea = document.getElementById('contestantsArea');
const noContestants = document.getElementById('noContestants');

// Modal refs
const contModal = document.getElementById('contModal');
const closeContModal = document.getElementById('closeContModal');
const modalTitle = document.getElementById('modalTitle');

const c_name = document.getElementById('c_name');
const c_language = document.getElementById('c_language');
const c_show_type = document.getElementById('c_show_type');
const c_season_no = document.getElementById('c_season_no');
const c_description = document.getElementById('c_description');
const c_photo_input = document.getElementById('c_photo_input');
const c_photo_preview = document.getElementById('c_photo_preview');
const c_photo_placeholder = document.getElementById('c_photo_placeholder');
const c_social_instagram = document.getElementById('c_social_instagram');
const c_social_facebook = document.getElementById('c_social_facebook');
const c_social_twitter = document.getElementById('c_social_twitter');
const saveContBtn = document.getElementById('saveContBtn');

// state
let contestants = [];
let languages = [];
let editingContestantId = null;
let contestantPhotoUrl = "";

/******** Helpers ********/
function escapeHtml(s){ if(!s) return ""; return String(s).replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;"); }
async function fetchJSON(url, opts={}){ const res = await fetch(url, opts); return res.json(); }

/******** Load languages ********/
async function loadLanguages(){
  try{
    const res = await fetchJSON(API_LANG);
    if(res && res.status === "success"){
      languages = res.data || [];
    } else {
      // fallback minimal set
      languages = [{id:1,name:"Malayalam"},{id:2,name:"Hindi"},{id:3,name:"Tamil"},{id:4,name:"English"}];
    }
    renderLanguageOptions();
  }catch(e){
    console.error("loadLanguages err", e);
    languages = [{id:1,name:"Malayalam"},{id:2,name:"Hindi"},{id:3,name:"Tamil"},{id:4,name:"English"}];
    renderLanguageOptions();
  }
}

function renderLanguageOptions(){
  filterLanguage.innerHTML = '<option value="">All Languages</option>';
  c_language.innerHTML = '<option value="">-- Select language --</option>';
  languages.forEach(l=>{
    const name = l.name || l;
    filterLanguage.innerHTML += `<option value="${escapeHtml(name)}">${escapeHtml(name)}</option>`;
    c_language.innerHTML += `<option value="${escapeHtml(name)}">${escapeHtml(name)}</option>`;
  });
}

/******** Load contestants ********/
async function loadContestantsFromServer(){
  try{
    const res = await fetchJSON(API_GET_CONTESTANTS);
    if(res && res.status === "success"){
      contestants = res.data || [];
    } else {
      contestants = [];
    }
    renderContestants();
  }catch(e){
    console.error("loadContestants err", e);
    contestants = [];
    renderContestants();
  }
}

/******** Render contestants list ********/
function renderContestants(){
  contestantsArea.innerHTML = "";
  const q = (searchBox.value || "").toLowerCase();
  const langFilter = (filterLanguage.value || "").trim();

  const list = contestants.filter(c=>{
    if(langFilter && String(c.language || "") !== langFilter) return false;
    if(!q) return true;
    const text = ((c.name||"") + " " + (c.language||"") + " " + (c.show_type||"") + " " + (c.season_no||"")).toLowerCase();
    return text.includes(q);
  });

  if(!list.length){
    noContestants.style.display = "block";
    return;
  }
  noContestants.style.display = "none";

  list.forEach(c=>{
    const card = document.createElement('div');
    card.className = "cont-card";

    const imgWrap = document.createElement('div');
    imgWrap.className = "cont-img";
    const img = document.createElement('img');
    img.src = c.photo || "";
    img.onerror = function(){ this.style.display = 'none'; };
    imgWrap.appendChild(img);

    const main = document.createElement('div');
    main.className = "cont-main";
    const nameEl = document.createElement('div');
    nameEl.className = "cont-name";
    nameEl.innerHTML = escapeHtml(c.name || "(no name)");

    const meta = document.createElement('div');
    meta.className = "cont-meta";
    meta.innerHTML = `<div>Language: <strong>${escapeHtml(c.language||"-")}</strong></div>
                      <div>Show: <strong>${escapeHtml(c.show_type||"-")}</strong></div>
                      <div>Season: <strong>${escapeHtml(c.season_no||"-")}</strong></div>`;

    const actions = document.createElement('div');
    actions.style.display = "flex";
    actions.style.gap = "8px";
    actions.style.justifyContent = "flex-end";

    const viewBtn = document.createElement('button');
    viewBtn.className = "btn btn-ghost";
    viewBtn.textContent = "View";
    viewBtn.onclick = ()=> openViewContestant(c.id);

    const editBtn = document.createElement('button');
    editBtn.className = "btn btn-ghost";
    editBtn.textContent = "Edit";
    editBtn.onclick = ()=> openEditContestant(c.id);

    const delBtn = document.createElement('button');
    delBtn.className = "btn btn-danger";
    delBtn.textContent = "Delete";
    delBtn.onclick = ()=> removeContestant(c.id);

    actions.appendChild(viewBtn);
    actions.appendChild(editBtn);
    actions.appendChild(delBtn);

    main.appendChild(nameEl);
    main.appendChild(meta);
    main.appendChild(actions);

    card.appendChild(imgWrap);
    card.appendChild(main);

    contestantsArea.appendChild(card);
  });
}

/******** Open view (simple alert modal) ********/
function openViewContestant(id){
  const c = contestants.find(x=>String(x.id) === String(id));
  if(!c) return alert("Not found");
  const social = JSON.parse(c.social || "[]") || {};
  const text = `Name: ${c.name}\nLanguage: ${c.language}\nShow: ${c.show_type}\nSeason: ${c.season_no}\n\nDescription:\n${c.description || "-"}\n\nSocial:\nInstagram: ${social.instagram || "-"}\nFacebook: ${social.facebook || "-"}\nTwitter: ${social.twitter || "-"}`;
  alert(text);
}

/******** Open create / edit modal ********/
createBtn.onclick = () => {
  editingContestantId = null;
  modalTitle.textContent = "Create Contestant";
  c_name.value = ""; c_language.value = ""; c_show_type.value = "season"; c_season_no.value = ""; c_description.value = "";
  c_photo_preview.style.display = "none"; c_photo_placeholder.style.display = "block"; contestantPhotoUrl = "";
  c_social_instagram.value = ""; c_social_facebook.value = ""; c_social_twitter.value = "";
  openModalCont();
};

function openModalCont(){ contModal.classList.add("open"); document.body.style.overflow = "hidden"; }
function closeModalCont(){ contModal.classList.remove("open"); document.body.style.overflow = ""; }
closeContModal.onclick = closeModalCont;
contModal.onclick = (e)=> { if(e.target === contModal) closeModalCont(); };

/******** Photo upload handler ********/
c_photo_input.onchange = async (ev) => {
  const f = ev.target.files[0];
  if(!f) return;
  try{
    const fd = new FormData();
    fd.append("file", f);
    const res = await fetch(API_UPLOAD_IMAGE, { method: "POST", body: fd });
    const j = await res.json();
    if(j.status === "success"){
      contestantPhotoUrl = j.url || j.url || ("uploads/" + j.url);
      c_photo_preview.src = contestantPhotoUrl;
      c_photo_preview.style.display = "block";
      c_photo_placeholder.style.display = "none";
    } else {
      alert("Upload failed: " + (j.message || ""));
    }
  }catch(e){
    console.error("upload err", e);
    alert("Upload failed");
  }
};

/******** Save contestant ********/
saveContBtn.onclick = async () => {
  const name = c_name.value.trim();
  const language = c_language.value.trim();
  const show_type = c_show_type.value.trim();
  const season_no = c_season_no.value.trim();
  const description = c_description.value.trim();
  const social = {
    instagram: c_social_instagram.value.trim(),
    facebook: c_social_facebook.value.trim(),
    twitter: c_social_twitter.value.trim()
  };

  if(!name){
    alert("Enter name");
    return;
  }

  const payload = {
    name, language, show_type, season_no, description,
    photo: contestantPhotoUrl || "",
    social
  };

  try{
    let res;
    if(editingContestantId){
      payload.id = editingContestantId;
      res = await fetchJSON(API_UPDATE_CONTESTANT, {
        method: "POST",
        headers: {"Content-Type":"application/json"},
        body: JSON.stringify(payload)
      });
    } else {
      res = await fetchJSON(API_SAVE_CONTESTANT, {
        method: "POST",
        headers: {"Content-Type":"application/json"},
        body: JSON.stringify(payload)
      });
    }

    if(res.status === "success"){
      closeModalCont();
      await loadContestantsFromServer();
    } else {
      alert("Error: " + (res.message || "Unknown"));
    }
  }catch(e){
    console.error("saveContestant err", e);
    alert("Save failed");
  }
};

/******** Edit contestant ********/
function openEditContestant(id){
  const c = contestants.find(x=>String(x.id) === String(id));
  if(!c) return alert("Not found");
  editingContestantId = id;
  modalTitle.textContent = "Edit Contestant";

  c_name.value = c.name || "";
  c_language.value = c.language || "";
  c_show_type.value = c.show_type || "season";
  c_season_no.value = c.season_no || "";
  c_description.value = c.description || "";

  contestantPhotoUrl = c.photo || "";
  if(contestantPhotoUrl){
    c_photo_preview.src = contestantPhotoUrl;
    c_photo_preview.style.display = "block";
    c_photo_placeholder.style.display = "none";
  } else {
    c_photo_preview.style.display = "none";
    c_photo_placeholder.style.display = "block";
  }

  try{
    const soc = JSON.parse(c.social || "[]") || {};
    c_social_instagram.value = soc.instagram || "";
    c_social_facebook.value = soc.facebook || "";
    c_social_twitter.value = soc.twitter || "";
  }catch(e){
    c_social_instagram.value = "";
    c_social_facebook.value = "";
    c_social_twitter.value = "";
  }

  openModalCont();
}

/******** Delete contestant ********/
async function removeContestant(id){
  if(!confirm("Delete contestant?")) return;
  try{
    const res = await fetchJSON(API_DELETE_CONTESTANT, {
      method: "POST",
      headers: {"Content-Type":"application/json"},
      body: JSON.stringify({id})
    });
    if(res.status === "success"){
      await loadContestantsFromServer();
    } else {
      alert("Delete failed: " + (res.message || ""));
    }
  }catch(e){
    console.error("delete err", e);
    alert("Delete failed");
  }
}

/******** Search / Filter wiring ********/
searchBtn.onclick = () => renderContestants();
filterLanguage.onchange = () => renderContestants();
searchBox.onkeyup = (e) => { if(e.key === "Enter") renderContestants(); };

/******** Init ********/
(async function init(){
  await loadLanguages();
  await loadContestantsFromServer();
})();
